/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2005 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2004-2005
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MSequences
//
//  This class describes a collection of sequences.
//
//  Such an input file looks like:
//
//     crab.seq:
//     ---------
//       AnalysisNumber: 1
//
//       SequencesOn: 35222
//       SequencesOff: 36817
//
//       Sequence00035222.File: sequences/sequence035222.txt
//       Sequence00036817.File: sequences/sequence036817.txt
//
//       Sequence00035222.Dir: /data2/wuerzburg/Crab-Analyse/images/035222
//       Sequence00036817.Dir: /data2/wuerzburg/Crab-Analyse/images/036817
//
// The analysis number is an artifical number used to name the output
// files automatically if the names are not overwritten in the corresponding
// programs.
//
// The sequence number are used to concatenate the filenames of the
// sequences using the file structure used in the datacenter.
//
// If you have different file names you can overwrite the default file names
// using Sequence%08d.File (make sure you have 8 digits!)
//
// In standard coditions (datacenter file system) paths are concatenated
// by using the information in the sequence files (date, etc). You can
// overwrite the directories in which the sequence-files (eg I-files) are
// stored using Sequence%08d.Dir (make sure you have 8 digits!)
//
// Resource file entries are case sensitive!
//
// MISSING (27/01/04): The default name and paths cannot be used yet, because
//                     they have to be defined soon.
//
/////////////////////////////////////////////////////////////////////////////
#include "MSequences.h"

#include <stdlib.h>

#include <TEnv.h>
#include <TRegexp.h>
#include <TSystem.h> // TSystem::ExpandPath

#include "MLog.h"
#include "MLogManip.h"

#include "MRead.h"
#include "MDirIter.h"
#include "MSequence.h"

ClassImp(MSequences);

using namespace std;

// --------------------------------------------------------------------------
//
// Copy the run numbers from the TString runs into the TArrayI data
//
void MSequences::Split(TString &runs, TArrayI &data) const
{
    const TRegexp regexp("[0-9]+");

    data.Set(0);
    runs = runs.Strip(TString::kTrailing);

    while (!runs.IsNull())
    {
        TString num = runs(regexp);

        const Int_t n = data.GetSize();
        data.Set(n+1);
        data[n] = atoi(num.Data());

        runs.Remove(0, runs.First(num)+num.Length());
    }
}

void MSequences::ResolveSequences(TEnv &env, const TArrayI &num, TList &list) const
{
    for (int i=0; i<num.GetSize(); i++)
    {
        TString name = env.GetValue(Form("Sequence%08d.File", num[i]), "");
        TString dir  = env.GetValue(Form("Sequence%08d.Dir",  num[i]), "");

        if (name.IsNull())
        {
            // Replace with correct default name
            name = Form("/data2/wuerzburg/sequences/sequence%08d.txt", num[i]);
        }
        /*
        if (dir.IsNull())
        {
            // Replace with default dir
        }
        */

        if (gSystem->AccessPathName(name, kFileExists))
            gLog << warn << "WARNING - Sequence file '" << name << "' doesn't exist." << endl;

        if (gSystem->AccessPathName(dir, kFileExists))
            gLog << warn << "WARNING - Directory '" << dir << "' doesn't exist." << endl;

        list.Add(new TNamed(name, dir));
    }
}

// --------------------------------------------------------------------------
//
// Read the file fname as setup file for the sequence.
//
MSequences::MSequences(const char *fname)
{
    fName  = fname;

    const char *expname = gSystem->ExpandPathName(fname);

    fTitle = Form("Sequences contained in file %s", expname);

    TEnv env(expname);
    delete [] expname;

    TString str;

    fNumAnalysis = env.GetValue("AnalysisNumber",  -1);

    str = env.GetValue("SequencesOn", "");
    Split(str, fNumSequencesOn);
    str = env.GetValue("SequencesOff", "");
    Split(str, fNumSequencesOff);


    ResolveSequences(env, fNumSequencesOn,  fSequencesOn);
    ResolveSequences(env, fNumSequencesOff, fSequencesOff);



    //Print();
    /*
     GetFileNames(env, fSequencesOn);
     GetFileNames(env, fSequencesOff);
     */
}

// --------------------------------------------------------------------------
//
// Return '+' if both can be accessed, '-' otherwise.
//
void MSequences::PrintFile(const TObject &obj)
{
    const Bool_t access = !gSystem->AccessPathName(obj.GetName(), kFileExists) && !gSystem->AccessPathName(obj.GetTitle(), kFileExists) ? '+' : '-';
    gLog << " " << (access?"+":"-") << " " << obj.GetName() << " <" << obj.GetTitle() << ">" << endl;
}

// --------------------------------------------------------------------------
//
// Print the contents of the sequence
//
void MSequences::Print(Option_t *o) const
{
    gLog << all;
    if (!IsValid())
    {
        gLog << "Sequence: " << fName << " <invalid>" << endl;
        return;
    }
    gLog << "Analysis Number: " << fNumAnalysis << endl;
    gLog << "Sequences On:   ";
    for (int i=0; i<fNumSequencesOn.GetSize(); i++)
        gLog << " " << fNumSequencesOn[i];
    gLog << endl;
    gLog << "Sequences Off:  ";
    for (int i=0; i<fNumSequencesOff.GetSize(); i++)
        gLog << " " << fNumSequencesOff[i];
    gLog << endl;

    if (!TString(o).Contains("files", TString::kIgnoreCase))
        return;

    TObject *obj=0;

    gLog << endl;
    gLog << "On-Data Files:" << endl;
    TIter NextOn(&fSequencesOn);
    while ((obj=NextOn()))
        PrintFile(*obj);

    gLog << endl;
    gLog << "Off-Data Files:" << endl;
    TIter NextOff(&fSequencesOff);
    while ((obj=NextOff()))
        PrintFile(*obj);
}

Bool_t MSequences::AddSequencesToList(const TList &list, MRead &read, char *id, Bool_t raw)
{
    MDirIter files;

    TIter Next(const_cast<TList*>(&list));
    TObject *o=0;
    while ((o=Next()))
    {
        MSequence seq(o->GetName());
        if (!seq.IsValid())
        {
            gLog << warn << "WARNING - Sequence " << o->GetName() << " invalid!" << endl;
            return kFALSE;
        }

        const TString dir(o->GetTitle());
        seq.SetupDatRuns(files, dir.IsNull() ? 0 : dir.Data(), id, raw);
    }

    return read.AddFiles(files)>0;
}

Bool_t MSequences::AddFiles(MRead &read, char *id, Bool_t raw) const
{
    const Bool_t rc1 = AddFilesOff(read, id, raw);
    const Bool_t rc2 = AddFilesOn(read, id, raw);
    return rc1 && rc2;
}

Bool_t MSequences::AddFilesOn(MRead &read, char *id, Bool_t raw) const
{
    return AddSequencesToList(fSequencesOn, read, id, raw);
}

Bool_t MSequences::AddFilesOff(MRead &read, char *id, Bool_t raw) const
{
    return AddSequencesToList(fSequencesOff, read, id, raw);
}
