/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 12/2004 <mailto:markus@ifae.es>
!         
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
//////////////////////////////////////////////////////////////////////////////
//
//  MCalibCalcFromPast
//
//  Steers the occurrance of interlaced calibration events in one data run
// 
//  Input Containers:
//   MParList
//   MCalibrationIntensityChargeCam
//   MCalibrationIntensityRelTimeCam
//   MBadPixelsIntensityCam 
//
//  Output Containers:
//   MCalibrationIntensityChargeCam
//   MCalibrationIntensityRelTimeCam
//   MBadPixelsIntensityCam 
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibCalcFromPast.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawRunHeader.h"

#include "MHCalibrationCam.h"

#include "MCalibrationIntensityChargeCam.h"
#include "MCalibrationIntensityBlindCam.h"
#include "MCalibrationIntensityQECam.h"
#include "MCalibrationIntensityRelTimeCam.h"

#include "MBadPixelsIntensityCam.h"

#include "MCalibrationChargeCalc.h"
#include "MCalibrationRelTimeCalc.h"
#include "MCalibrateData.h"

ClassImp(MCalibCalcFromPast);

using namespace std;

const UInt_t MCalibCalcFromPast::fgNumEventsDump = 500;

// --------------------------------------------------------------------------
//
//  Default constructor. 
//
// Sets:
// - fNumEventsDump to fgNumEventsDump
//
MCalibCalcFromPast::MCalibCalcFromPast(const char *name, const char *title)
    : fGeom(NULL), fParList(NULL), fRunHeader(NULL),
      fIntensCharge(NULL), fIntensBlind(NULL), fIntensRelTime(NULL), fIntensBad(NULL),
      fChargeCalc(NULL), fRelTimeCalc(NULL), fCalibrate(NULL),
      fNumEvents(0), fNumCam(0)
{

  fName  = name  ? name  : "MCalibCalcFromPast";
  fTitle = title ? title : "Task to steer the processing of interlaced calibration events";

  SetNumEventsDump();
}

// -----------------------------------------------------------------------------------
//
Int_t MCalibCalcFromPast::PreProcess(MParList *pList)
{

  fGeom = (MGeomCam*)pList->FindObject("MGeomCam");
  if (!fGeom)
    {
      *fLog << err << "MGeomCam not found... abort." << endl;
      return kFALSE;
    }

  fParList = pList;
  if (!fParList)
    {
      *fLog << err << "MParList not found... abort." << endl;
      return kFALSE;
    }

  fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
  if (!fRunHeader)
  {
      *fLog << err << "MRawRunHeader not found... abort." << endl;
      return kFALSE;
  }

  // 
  // Look for the MBadPixels Intensity Cam
  //
  fIntensBad = (MBadPixelsIntensityCam*)pList->FindCreateObj("MBadPixelsIntensityCam");
  if (fIntensBad)
    *fLog << inf << "Found MBadPixelsIntensityCam ... " << flush;
  else
    return kFALSE;
  
  // 
  // Look for the MFillH-class "MHCalibrationBlindCam". In case yes, initialize the 
  // corresponding IntensityCam
  //
  if (pList->FindObject(AddSerialNumber("MHCalibrationBlindCam")))
  {

    *fLog << inf << "Found MHCalibrationBlindCam ... " << flush;

    fIntensBlind = (MCalibrationIntensityBlindCam*)pList->FindCreateObj("MCalibrationIntensityBlindCam");
    if (!fIntensBlind)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensitBlindCam abort... " << endl;
        return kFALSE;
      }
  }
  
  // 
  // Look for the MFillH-class "MHCalibrationChargeCam". In case yes, initialize the 
  // corresponding IntensityCam
  //
  if (pList->FindObject(AddSerialNumber("MHCalibrationChargeCam")))
  {

    fIntensCharge = (MCalibrationIntensityChargeCam*)pList->FindCreateObj("MCalibrationIntensityChargeCam");
    fIntensQE     = (MCalibrationIntensityQECam*)    pList->FindCreateObj("MCalibrationIntensityQECam");

    if (!fChargeCalc)
      fChargeCalc   = (MCalibrationChargeCalc*)pList->FindObject("MCalibrationChargeCalc");

    if (!fCalibrate)
      fCalibrate   = (MCalibrateData*)pList->FindObject("MCalibrateData");
    
    *fLog << inf << "Found MHCalibrationChargeCam ... " << flush;

    if (!fIntensCharge)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityChargeCam abort... " << endl;
        return kFALSE;
      }
    
    if (!fIntensQE)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityQECam abort... " << endl;
        return kFALSE;
      }
    
    if (!fChargeCalc)
      {
        *fLog << err << "Could not find MCalibrationChargeCalc abort... " << endl;
        return kFALSE;
      }

    if (!fCalibrate)
      {
        *fLog << err << "Could not find MCalibrateData abort... " << endl;
        return kFALSE;
      }
  }
      
  // 
  // Look for the MFillH name "FillRelTimeCam". In case yes, initialize the 
  // corresponding IntensityCam
  //
  if (pList->FindObject(AddSerialNumber("MHCalibrationRelTimeCam")))
  {

    fIntensRelTime = (MCalibrationIntensityRelTimeCam*)pList->FindCreateObj("MCalibrationIntensityRelTimeCam");
    if (!fRelTimeCalc)
      fRelTimeCalc   = (MCalibrationRelTimeCalc*)pList->FindObject(AddSerialNumber("MCalibrationRelTimeCalc"));

    *fLog << inf << "Found MHCalibrationRelTimeCam ... " << flush;

    if (!fIntensRelTime)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityRelTimeCam abort... " << endl;
        return kFALSE;
      }

    if (!fRelTimeCalc)
      {
        *fLog << err << "Could not find MCalibrationRelTimeCalc abort... " << endl;
        return kFALSE;
      }
  }
      
  fNumCam    = 0;
  fNumEvents = 0;

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// - Initializes new containers in the 
// - Intensity Cams, if the number of calibration events has reach fNumEventsDump. 
// - Executes Finalize() of the MCalibration*Calc classes in that case.
// - Sets the latest MCalibrationChargeCam as update class into MCalibrateData
// - Initialize new MCalibration*Cams into the intensity cams.
//
Int_t MCalibCalcFromPast::Process()
{

  if (fNumEvents++ < fNumEventsDump)
    return kTRUE;
  
  fNumEvents = 0;
  //
  // Finalize Possible calibration histogram classes...
  //
  *fLog << inf << GetDescriptor() << ": Finalize calibration histograms: " << flush;

  if (Finalize("MHCalibrationChargeCam"))      *fLog << "MHCalibrationChargeCam..." << flush;
  if (Finalize("MHCalibrationChargeBlindCam")) *fLog << "MHCalibrationChargeBlindCam..." << flush;
  if (Finalize("MHCalibrationRelTimeCam"))     *fLog << "MHCalibrationRelTimeCam..." << flush;

  //
  // Finalize possible calibration calculation tasks
  //
  *fLog << endl;
  *fLog << inf << "Finalize calibration calculations..." << endl;
  if (fChargeCalc)
    fChargeCalc->Finalize();
  if (fRelTimeCalc)
    fRelTimeCalc->Finalize();
  if (fCalibrate)
    fCalibrate->UpdateConversionFactors((MCalibrationChargeCam*)fIntensCharge->GetCam());

  ReInitialize();
  return kTRUE;
}


// --------------------------------------------------------------------------
//
// Searches for name in the MParList and calls, if existing: 
// - MHCalibrationCam::Finalize()
// - MHCalibrationCam::ResetHists()
//
Bool_t MCalibCalcFromPast::Finalize(const char* name)
{

  MHCalibrationCam *hist = (MHCalibrationCam*)fParList->FindObject(name);
  if (hist)
    {
      hist->Finalize();
      hist->ResetHists();
      return kTRUE;
    }

  return kFALSE;
  
}

// --------------------------------------------------------------------------
//
// Re-Intitializes new containers inside the Intensity Cams. 
// From now on, a call to the IntensityCam functions returns pointers 
// to the newly created Containers.
//
Bool_t MCalibCalcFromPast::ReInitialize()
{
  fNumCam++;

  *fLog << inf << "MCalibCalcFromPast::ReInitialize #" << fNumCam << " ";

  const Int_t runnumber = fRunHeader->GetRunNumber();

  if (fIntensBad)
    {
      fIntensBad->AddToList(Form("MBadPixelsCam%04d",fNumCam),*fGeom);
      *fLog << "MBadPixelsCam...";
    }

  if (fIntensCharge)
    {
      fIntensCharge->AddToList(Form("MCalibrationChargeCam%04d",fNumCam),*fGeom);
      fIntensCharge->GetCam()->SetRunNumber(runnumber);
      *fLog << "MCalibrationChargeCam...";
    }
  if (fIntensQE)
    {
      fIntensQE->AddToList(Form("MCalibrationQECam%04d",fNumCam),*fGeom);
      fIntensQE->GetCam()->SetRunNumber(runnumber);
      *fLog << "MCalibrationQECam...";
    }
  if (fIntensBlind)
    {
      fIntensBlind->AddToList(Form("MCalibrationBlindCam%04d",fNumCam),*fGeom);
      fIntensBlind->GetCam()->SetRunNumber(runnumber);
      *fLog << "MCalibrationBlindCam...";
    }

  *fLog << endl;

  return kTRUE;

}

Int_t MCalibCalcFromPast::PostProcess()
{
  *fLog << "Number of Calibration Cams: " << fNumCam << endl;
  return kTRUE;
  
}

