/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MEvtLoop                                                                //
//                                                                         //
// This class is the core of each event processing.                        //
// First you must set the parameter list to use. The parameter list        //
// must contain the task list (MTaskList) to use. The name of the task     //
// list can be specified if you call Eventloop. The standard name is       //
// "MTaskList". The name you specify must match the name of the MTaskList  //
// object.                                                                 //
//                                                                         //
// If you call Eventloop first all PreProcess functions - with the         //
// parameter list as an argument - of the tasks in the task list are       //
// executed. If one of them returns kFALSE then the execution is stopped.  //
// If the preprocessing was ok. The Process funtion of the tasks are       //
// as long as one function returns kSTOP. Only the tasks which are marked  //
// marked as "All" or with a string which matches the MInputStreamID of    //
// MTaskList are executed. If one tasks returns kCONTINUE the pending      //
// tasks in the list are skipped and the execution in continued with       //
// the first one in the list.                                              //
// Afterwards the PostProcess functions are executed.                      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MEvtLoop.h"

#include <iostream.h>

#include <TStopwatch.h>

#include "MLog.h"
#include "MParList.h"
#include "MTaskList.h"

ClassImp(MEvtLoop)

MEvtLoop::MEvtLoop() : fParList(NULL)
{
}

MEvtLoop::~MEvtLoop()
{
}

Bool_t MEvtLoop::PreProcess(const char *tlist)
{
    //
    // The proprocessing part of the eventloop. Be careful, this is
    // for developers use only!
    //

    //
    // check if the needed parameter list is set.
    //
    if (!fParList)
    {
        *fLog << "MEvtLoop::Eventloop - Error: Parlist not initialized." << endl;
        return kFALSE;
    }

    //
    //  check for the existance of the specified task list
    //  the default name is "MTaskList"
    //
    fTaskList = (MTaskList*)fParList->FindObject(tlist);
    if (!fTaskList)
    {
        *fLog << "MEvtLoop::Eventloop - Error: Cannot find tasklist '" << tlist << "' in parameter list." << endl;
        return kFALSE;
    }

    if (fLog != &gLog)
    {
        fParList ->SetLogStream(fLog);
        fTaskList->SetLogStream(fLog);
    }

    //
    //  execute the preprocess of all tasks
    //  connect the different tasks with the right containers in
    //  the parameter list
    //
    if (!fTaskList->PreProcess(fParList))
    {
        *fLog << "Error detected while PreProcessing" << endl;
        return kFALSE;
    }

    *fLog << endl;

    return kTRUE;
}

void MEvtLoop::Process(Int_t maxcnt) const
{
    //
    // The processing part of the eventloop. Be careful, this is
    // for developers use only!
    //

    //
    //   loop over all events and process all tasks for
    //   each event
    //
    *fLog << "Eventloop running (";

    if (maxcnt<0)
        *fLog << "all";
    else
        *fLog << dec << maxcnt;

    *fLog << " events)..." << flush;

    Int_t dummy = maxcnt<0 ? 0 : maxcnt;

    //
    // start a stopwatch
    //
    TStopwatch clock;
    clock.Start();

    //
    // This is the MAIN EVENTLOOP which processes the data
    // if maxcnt<0 the number of processed events is counted
    // else only maxcnt events are processed
    //
    if (maxcnt<0)
        while (fTaskList->Process() && ++dummy);
    else
        while (fTaskList->Process() && dummy--);

    //
    // stop stop-watch, print results
    //
    clock.Stop();

    *fLog << "Ready!" << endl << endl;

    clock.Print();

    *fLog << dec << endl
        << "Time: " << clock.CpuTime() << "s"
        << " for " << (maxcnt<0?dummy:maxcnt) << " Events"
        << " --> " << (maxcnt<0?dummy:maxcnt)/clock.CpuTime() << " Events/s"
        << endl << endl;

}

void MEvtLoop::PostProcess() const
{
    //
    // The postprocessing part of the eventloop. Be careful, this is
    // for developers use only!
    //

    //
    //  execute the post process of all tasks
    //
    fTaskList->PostProcess();
}

void MEvtLoop::Eventloop(Int_t maxcnt, const char *tlist)
{
    // See class description above.

    if (!PreProcess(tlist))
        return;

    Process(maxcnt);

    PostProcess();
}

