/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 08/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Daniela Dorner, 08/2004 <mailto:dorner@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// buildsequenceentries.C
// ======================
//
// to group the runs of one night into sequences, this marco:
// - reads the runinformation of one night from the database
// - sorts the runs by source
// - groups the runs to sequences such that each run belongs
//   to the nearest (in time) calibration run
//
// apart from that the runs in one sequence must have the same
// hv-settings, trigger tables, light conditions, DT tables,
// trigger delay tables and testflag
//
/////////////////////////////////////////////////////////////////////////////
#include <iostream>
#include <iomanip>
#include <fstream>

#include <MSQLServer.h>
#include <TSQLRow.h>
#include <TSQLResult.h>

#include <TEnv.h>
#include <TMath.h>
#include <TExMap.h>
#include <TArrayI.h>
#include <TRegexp.h>
#include <TSystem.h>

#include <MTime.h>
#include <MDirIter.h>

using namespace std;

int debug = 0;

Bool_t DeleteSequences(MSQLServer &serv, TString cond)
{
    TString query(Form("SELECT fSequenceFirst FROM Sequences"
                       " WHERE %s AND fManuallyChangedKEY=1", cond.Data()));

    TSQLResult *res = serv.Query(query);
    if (!res)
        return kFALSE;

    TSQLRow *row=0;
    while ((row=res->Next()))
    {
        query = Form("DELETE FROM Calibration WHERE fSequenceFirst=%s", (*row)[0]);

        TSQLResult *res = serv.Query(query);
        if (!res)
            return kFALSE;
        delete res;

        query = Form("DELETE FROM Star WHERE fSequenceFirst=%s", (*row)[0]);

        TSQLResult *res = serv.Query(query);
        if (!res)
            return kFALSE;
        delete res;

        query = Form("DELETE FROM SequenceProcessStatus WHERE fSequenceFirst=%s", (*row)[0]);

        res = serv.Query(query);
        if (!res)
            return kFALSE;
        delete res;

        Int_t sequno=atoi((*row)[0]);
        TString fname(Form("/mnt/stk01/sequences/%04d/sequence%08d.txt", sequno/10000, sequno));
        gSystem->Unlink(fname);

        query = Form("UPDATE RunData SET fSequenceFirst=0 WHERE fSequenceFirst=%s", (*row)[0]);

        res = serv.Query(query);
        if (!res)
            return kFALSE;
        delete res;

    }
    delete res;

    query = Form("DELETE FROM Sequences WHERE %s AND fManuallyChangedKEY=1", cond.Data());
    res = serv.Query(query);
    if (!res)
        return kFALSE;

    delete res;

    return kTRUE;
}

Int_t CheckOverlap(MSQLServer &serv, Int_t from, Int_t to)
{
    TString query(Form("SELECT fSequenceFirst FROM Sequences WHERE"
                       " (%d BETWEEN fSequenceFirst AND fSequenceLast) OR "
                       " (%d BETWEEN fSequenceFirst AND fSequenceLast)", from, to));

    TSQLResult *res = serv.Query(query);
    if (!res)
        return -1;

    TSQLRow *row = res->Next();

    Int_t rc = row ? kFALSE : kTRUE;
    if (rc==kFALSE)
        cout << "Sorry - the sequence from " << from << " to " << to << " overlaps with sequence #" << (*row)[0] << endl;

    delete res;

    return rc;
}

Int_t DoCheck(TSQLResult &res, Int_t from, Int_t to)
{
    TSQLRow *row=0;

    TArrayI data(9);

    Int_t n = 0;

    while ((row=res.Next()))
    {
        n++;

        if (data[0]==0)
        {
            for (int i=0; i<data.GetSize(); i++)
                data[i] = atoi((*row)[i]);
            continue;
        }

        for (int i=1; i<data.GetSize(); i++)
        {
            if (data[i] != atoi((*row)[i]))
                return i+1;
        }
    }
    return n==0 ? 0 : -1;
}

Bool_t CheckSequence(MSQLServer &serv, Int_t from, Int_t to, Int_t src, Int_t type)
{
    TString query("SELECT fRunNumber, fL1TriggerTableKEY, fL2TriggerTableKEY,"
                  " fProjectKEY, fHvSettingsKEY, fDiscriminatorThresholdTableKEY,"
                  " fTriggerDelayTableKEY, fLightConditionsKEY, fTestFlagKEY"
                  " FROM RunData");
    query += Form(" WHERE fRunTypeKEY=%d AND fSourceKEY=%d AND fExcludedFDAKEY=1 AND (fRunNumber BETWEEN %d AND %d)"
                  " ORDER BY fRunNumber", type, src, from, to);

    TSQLResult *res = serv.Query(query);
    if (!res)
        return kFALSE;

    Int_t rc = DoCheck(*res, from, to);
    delete res;

    switch (rc)
    {
    case 0: cout << "ERROR - No runs found for check!"                        << endl; break;
    case 1: cout << "ERROR - fRunNumber doesn't match!"                       << endl; break;
    case 2: cout << "ERROR - fL1TriggerTableKEY doesn't match!"                 << endl; break;
    case 3: cout << "ERROR - fL2TriggerTableKEY doesn't match!"                 << endl; break;
    case 4: cout << "ERROR - fProjectKEY doesn't match!"                      << endl; break;
    case 5: cout << "ERROR - fHvSettingsKEY doesn't match!"                   << endl; break;
    case 6: cout << "ERROR - fDiscriminatorThresholdTableKEY doesn't match!"  << endl; break;
    case 7: cout << "ERROR - fTriggerDelayTableKEY doesn't match!"            << endl; break;
    case 8: cout << "ERROR - fLightConditionsKEY doesn't match!"              << endl; break;
    case 9: cout << "ERROR - fTestFlagKEY doesn't match!"                     << endl; break;
    }

    return rc<0;
}

Bool_t InsertSequence(MSQLServer &serv, Int_t from, Int_t to, Int_t src, Bool_t dummy)
{
    Int_t rc = dummy ? kTRUE : CheckOverlap(serv, from, to);
    if (rc<=0)
        return kFALSE;

    // ========== Request number of events ==========
    TString query("SELECT SUM(fNumEvents), "
                  " SUM(if(TIME_TO_SEC(fRunStop)-TIME_TO_SEC(fRunStart)<0,"
                  "  TIME_TO_SEC(fRunStop)-TIME_TO_SEC(fRunStart)+24*60*60,"
                  "  TIME_TO_SEC(fRunStop)-TIME_TO_SEC(fRunStart))), ");
    query +=      " MIN(fZenithDistance), MAX(fZenithDistance), ";
    query +=      " MIN(fAzimuth), MAX(fAzimuth) ";
    query += Form(" FROM RunData"
                  " WHERE fRunTypeKEY=2 AND fSourceKEY=%d AND (fRunNumber BETWEEN %d AND %d) AND fExcludedFDAKEY=1",
                  src, from, to);

    TSQLResult *res = serv.Query(query);
    if (!res)
        return kFALSE;

    TSQLRow *row = res->Next();
    if (!row || !(*row)[0])
    {
        cout << "ERROR - No result from query: " << query << endl;
        return kFALSE;
    }

    TString nevts = (*row)[0];
    TString secs  = (*row)[1];
    TString zdmin = (*row)[2];
    TString zdmax = (*row)[3];
    TString azmin = (*row)[4];
    TString azmax = (*row)[5];

    delete res;

    // ========== Request start time of sequence ==========
    query = Form("SELECT fRunStart FROM RunData WHERE fRunNumber=%d AND fSourceKEY=%d AND fExcludedFDAKEY=1", from, src);

    res = serv.Query(query);
    if (!res)
        return kFALSE;

    row = res->Next();
    if (!row || !(*row)[0])
    {
        cout << "ERROR - No result from query: " << query << endl;
        return kFALSE;
    }

    TString start((*row)[0]);

    delete res;

    // ========== Request data of sequence ==========
    query = Form("SELECT fProjectKEY, fL1TriggerTableKEY, fL1TriggerTableKEY,"
                 " fHvSettingsKEY, fDiscriminatorThresholdTableKEY,"
                 " fTriggerDelayTableKEY, fLightConditionsKEY, fTestFlagKEY"
                 " FROM RunData"
                 " WHERE fRunTypeKEY=2 AND fSourceKEY=%d AND fExcludedFDAKEY=1 AND (fRunNumber BETWEEN %d AND %d)"
                 " LIMIT 1", src, from, to);

    res = serv.Query(query);
    if (!res)
        return kFALSE;

    row = res->Next();
    if (!row)
    {
        cout << "ERROR - No result from query: " << query << endl;
        return kFALSE;
    }

    TString query1 = Form("INSERT Sequences SET"
                          " fSequenceFirst=%d,"
                          " fSequenceLast=%d,"
                          " fProjectKEY=%s,"
                          " fSourceKEY=%d,"
                          " fNumEvents=%s,"
                          " fRunTime=%s,"
                          " fRunStart=\"%s\","
                          " fZenithDistanceMin=%s,"
                          " fZenithDistanceMax=%s,"
                          " fAzimuthMin=%s,"
                          " fAzimuthMax=%s,"
                          " fL1TriggerTableKEY=%s,"
                          " fL2TriggerTableKEY=%s,"
                          " fHvSettingsKEY=%s,"
                          " fDiscriminatorThresholdTableKEY=%s,"
                          " fTriggerDelayTableKEY=%s,"
                          " fLightConditionsKEY=%s,"
                          " fTestFlagKEY=%s,"
                          " fManuallyChangedKEY=1",
                          from, to, (*row)[0], src, nevts.Data(),
                          secs.Data(), start.Data(), zdmin.Data(),
                          zdmax.Data(), azmin.Data(), azmax.Data(),
                          (*row)[1], (*row)[2], (*row)[3],
                          (*row)[4], (*row)[5], (*row)[6],
                          (*row)[7]);

    TString query2 = Form("UPDATE RunData SET fSequenceFirst=%d WHERE"
                          "  (fRunNumber  BETWEEN %d AND %d) AND"
                          "  (fRunTypeKEY BETWEEN  2 AND  4) AND"
                          "  fSourceKEY=%d AND fHvSettingsKEY=%s AND fExcludedFDAKEY=1",
                          from, from, to, src, (*row)[3]);

    TString query3 = Form("INSERT SequenceProcessStatus SET fSequenceFirst=%d ",
                          from);
    delete res;

    if (dummy)
        return kTRUE;

    res = serv.Query(query1);
    if (!res)
        return kFALSE;
    delete res;

    res = serv.Query(query2);
    if (!res)
        return kFALSE;
    delete res;

    res = serv.Query(query3);
    if (!res)
        return kFALSE;
    delete res;

    return kTRUE;
}

Bool_t NewSequence(MSQLServer &serv, Int_t from, Int_t to, Int_t src, Bool_t dummy)
{
    cout << "Found Sequence (" << from << ", " << to << ") ... checking..." << flush;

    if (!CheckSequence(serv, from, to, src, 2))
    {
        cout << "Warning - Found inconsistency in data-runs (" << from << ", " << to << ")" << endl;
        //sequence is not built, but kTRUE is returned, to allow
        //the automatic processing of the other sequences of this day
        return kTRUE;
    }
    if (!CheckSequence(serv, from, to, src, 3))
    {
        cout << "Warning - Found inconsistency in ped-runs (" << from << ", " << to << ")" << endl;
        //sequence is not built, but kTRUE is returned, to allow
        //the automatic processing of the other sequences of this day
        return kTRUE;
    }

    cout << "ok." << endl;

    Bool_t rc = InsertSequence(serv, from, to, src, dummy);
    if (!rc)
        cout << "InsertSequence failed!" << endl;

    return rc;
}

Bool_t GetSources(MSQLServer &serv, TString cond, TArrayI &srcs)
{
    TString query(Form("SELECT fSourceKEY"
                       " FROM RunData"
                       " WHERE %s AND fExcludedFDAKEY=1 AND (fRunTypeKEY BETWEEN 2 AND 4) GROUP BY fSourceKEY",
                       cond.Data())
                 ); //DATE(fStartTime)=\"2004-05-19\"");

    TSQLResult *res = serv.Query(query);
    if (!res)
        return kFALSE;

    srcs.Set(res->GetRowCount());

    cout << "Found " << srcs.GetSize() << " sources." << endl << endl;

    TSQLRow *row=0;
    Int_t i=0;
    while ((row=res->Next()))
        srcs[i++] = atoi((*row)[0]);

    delete res;
    return kTRUE;
}

Bool_t Process(MSQLServer &serv, TString cond, Int_t src, Bool_t dummy)
{
    if (debug)
        cout << "Processing Source: " << src << endl;

    TString query(Form("SELECT fRunNumber, fRunTypeKEY, fRunStart, fRunStop"
                       " FROM RunData"
                       " WHERE %s AND fSourceKEY=%d AND fExcludedFDAKEY=1 AND"
                       " (fRunTypeKEY BETWEEN 2 AND 4)"
                       " ORDER BY fRunNumber", cond.Data(), src)
                 ); //DATE(fStartTime)=\"2004-05-19\"");

    TSQLResult *res = serv.Query(query);
    if (!res)
        return kFALSE;

    TExMap map;

    Int_t start=0;
    Int_t stop=0;
    Int_t last=0;
    Int_t first=0;

    MTime lasttime;

    TSQLRow *row=0;

    enum { UNKNOWN, PED=3, CAL=4, DATA=2 };
    Char_t status = UNKNOWN;

    Int_t nblocks = 0;

    while ((row=res->Next()))
    {
        if (!(*row)[1])
            continue;

        if (start==0)
        {
            first = atoi((*row)[0]);
            if (debug)
                cout << "First Run: " << first << endl;
        }

        switch (atoi((*row)[1]))
        {
        case CAL: // ---------- CALIBRATION ----------
            if (status!=CAL)
            {
                start = stop = atoi((*row)[0]);
                if (!(*row)[2])
                    cout << "No time available... skipped." << endl;
                else
                {
                    MTime *tm = new MTime;
                    tm->SetSqlDateTime((*row)[2]);
                    map.Add((ULong_t)map.GetSize(), (Long_t)nblocks, (Long_t)tm);
                }
            }
            status = CAL;
            break;
        default:
            if (status==CAL)
            {
                MTime *tm = new MTime(lasttime);
                map.Add((ULong_t)map.GetSize(), (Long_t)nblocks, (Long_t)tm);

                stop = last;
                nblocks++;
                if (debug)
                    cout << "Cal Block #" << nblocks << " from " << start << " to " << last << endl;
            }
            status = UNKNOWN;
            break;
        }
        last = atoi((*row)[0]);
        lasttime.SetSqlDateTime((*row)[3]);
    }
    if (status==CAL)
    {
        stop = last;
        nblocks++;
        if (debug)
            cout << "Cal Block #" << nblocks << " from " << start << " to " << stop << endl;
    }

    if (debug)
        cout << "Last Run: " << last << endl;
    delete res;

    if (debug)
        cout << "Found " << nblocks << " calibration blocks" << endl;

    res = serv.Query(query);
    if (!res)
        return kFALSE;

    Int_t n = -1;

    Bool_t rc = kTRUE;

    start = first;
    while ((row=res->Next()))
    {
        if (!(*row)[1])
            continue;

        MTime tstart, tstop;
        tstart.SetSqlDateTime((*row)[2]);
        tstop.SetSqlDateTime((*row)[3]);

        MTime    min;
        Int_t    nmin = -1;
        Double_t dmin = 1e35;

        Long_t key, val;
        TExMapIter nmap(&map);
        while (nmap.Next(key, val))
        {
            MTime *t = (MTime*)val;

            if (nmin==-1)
            {
                nmin = key;
                min  = *(MTime*)val;
                dmin = fabs((Double_t)*t-(Double_t)tstart);
            }

            if (fabs((Double_t)*t-(Double_t)tstart) < dmin)
            {
                nmin = key;
                dmin = fabs((Double_t)*t-(Double_t)tstart);
                min = *t;
            }
            if (fabs((Double_t)*t-(Double_t)tstop) < dmin)
            {
                nmin = key;
                dmin = fabs((Double_t)*t-(Double_t)tstop);
                min = *t;
            }
        }

        if (n!=nmin)
        {
            if (n!=-1)
            {
                if (!NewSequence(serv, start, last, src, dummy))
                {
                    rc = kFALSE;
                    //continue;
                }
            }
            n = nmin;
            start = atoi((*row)[0]);
        }
        last = atoi((*row)[0]);
    }

    delete res;

    if (n!=-1 && start!=last)
    {
        if (!NewSequence(serv, start, last, src, dummy))
            rc = kFALSE;
    }

    if (debug)
        cout << endl;

    return rc; 
}

// This tool will work from Period017 (2004_05_17) on...
int buildsequenceentries(TString day, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLServer serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return 0;
    }

    cout << "buildsequences" << endl;
    cout << "--------------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "Night of sunrise at: " << day << endl;
    cout << endl;

    day += " 13:00:00";
    const TString cond(Form("(fRunStart>ADDDATE(\"%s\", INTERVAL -1 DAY) AND fRunStart<\"%s\")",
                            day.Data(), day.Data()));

    if (!dummy && !DeleteSequences(serv, cond))
        return 0;

    Bool_t rc = kTRUE;

    // get all sources for this day
    TArrayI src;
    GetSources(serv, cond, src);
    // find and build the sequences for the day for each source
    for (int i=0; i<src.GetSize(); i++)
        if (!Process(serv, cond, src[i], dummy))
            rc = kFALSE;

    return rc ? 1 : 0;
}
