/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 08/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Daniela Dorner, 08/2004 <mailto:dorner@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2006
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// filldotraw.C
// ============
//
// This macro is used to read a merpped raw data file or a raw data file
// directly. The descision is taken by the file-name extension (".root" or
// ".raw")
//
// Usage:
//   .x filldotraw.C("/data/MAGIC/Period014/filename.raw", kTRUE)
//
// The second argument is the 'dummy-mode'. If it is kTRUE dummy-mode is
// switched on and nothing will be written into the database. This is usefull
// for tests.
//
// Filling the database is done with 'UPADTE' for _all_ columns
// matching the Run-Number!
//
// The macro can also be run without ACLiC but this is a lot slower...
//
// Remark: Running it from the commandline looks like this:
//   root -q -l -b filldotraw.C+\(\"filename\"\,kFALSE\) 2>&1 | tee filldotraw.log
//
// Make sure, that database and password are corretly set in a resource
// file called sql.rc and the resource file is found.
//
// Returns 2 in case of failure, 1 in case of success and 0 if the connection
// to the database is not working.
//
/////////////////////////////////////////////////////////////////////////////

#include <fstream>
#include <iostream>

#include <TEnv.h>
#include <TFile.h>
#include <TTree.h>
#include <TRegexp.h>

#include <TSQLRow.h>
#include <TSQLResult.h>

#include "MZlib.h"
#include "MSQLServer.h"
#include "MRawRunHeader.h"
#include "MDirIter.h"

using namespace std;

//get key for a magic number
Int_t MagicNumber(MSQLServer &serv, const MRawRunHeader &h)
{
    TString query(Form("SELECT fMagicNumberKEY FROM MagicNumber WHERE fMagicNumber=%d",
                       h.GetMagicNumber()));

    TSQLResult *res = serv.Query(query);
    if (!res)
    {
        cout << "ERROR - Query failed: " << query << endl;
        return -1;
    }

    TSQLRow *row = res->Next();

    TString rc = row ? (*row)[0] : "";

    delete res;

    if (rc.IsNull())
    {
        cout << "ERROR - No result from query: " << query << endl;
        return -1;
    }

    return rc.Atoi();
}

Bool_t ReadRaw(TString fname, MRawRunHeader &h)
{
    MZlib fin(fname);
    if (!fin)
    {
        cout << "ERROR - Couldn't open file " << fname << endl;
        return kFALSE;
    }

    if (!h.ReadEvt(fin))
    {
        cout << "ERROR - Reading header from file " << fname << endl;
        return kFALSE;
    }
    return kTRUE;
}

Bool_t ReadRoot(TString fname, MRawRunHeader *h)
{
    TFile file(fname, "READ");
    if (file.IsZombie())
    {
        cout << "ERROR - Cannot open file " << fname << endl;
        return kFALSE;
    }

    TTree *t = (TTree*)file.Get("RunHeaders");
    if (!t)
    {
        cout << "ERROR - Tree RunHeaders not found." << endl;
        return kFALSE;
    }

    t->SetBranchAddress("MRawRunHeader.", &h);
    t->GetEntry(0);

    return kTRUE;
}

int Process(MSQLServer &serv, TString fname, Bool_t dummy)
{
    MRawRunHeader h;

    //read header either from root or from raw file
    if (fname.EndsWith(".root"))
        ReadRoot(fname, &h);
    if (fname.EndsWith(".raw"))
        ReadRaw(fname, h);
    if (fname.EndsWith(".raw.gz"))
        ReadRaw(fname, h);

    if (dummy)
    {
        h.Print("header");
        return 1;
    }

    //get key for the magic number
    const Int_t key = MagicNumber(serv, h);
    if (key<0)
        return 2;

    TString query(Form("UPDATE RunData SET fMagicNumberKEY=%d, fFormatVersion=%d WHERE fRunNumber=%d",
                       key, h.GetFormatVersion(), h.GetRunNumber()));

    TSQLResult *res = serv.Query(query);
    if (!res)
    {
        cout << "ERROR - Query failed: " << query << endl;
        return 2;
    }
    delete res;
    return 1;
}

int filldotraw(TString fname, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLServer serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return 0;
    }

    cout << "filldotraw" << endl;
    cout << "----------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "File: " << fname << endl;
    cout << endl;

    return Process(serv, fname, dummy);
}

int filldotraw(Int_t runno, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLServer serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return 0;
    }

    cout << "filldotraw" << endl;
    cout << "----------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "Run: " << runno << endl;
    cout << endl;

    //get date for the run to build path of the file
    TString query(Form("SELECT DATE_FORMAT(ADDDATE(fRunStart, Interval 13 HOUR), '%%Y/%%m/%%d') FROM RunData WHERE fRunNumber=%d",
                       runno));

    TSQLResult *res = serv.Query(query);
    if (!res)
    {
        cout << "ERROR - Query failed: " << query << endl;
        return 2;
    }

    TSQLRow *row = 0;
    row = res->Next();
    TString date=(*row)[0];
    cout << "date: " << date << endl;
    TString path(Form("/magic/data/rawfiles/%s", date.Data()));
    TString file(Form("*%d_*_*_E.raw.?g?z?", runno));

    delete res;

    cout << "path: " << path << " - file : " << file << endl;
    TString fname;
    TString name;

    Int_t count=0;
    MDirIter Next(path, file, -1);
    while (1)
    {
        name = Next();
        if (name.IsNull())
            break;
        fname=name;
        cout << "filename: " << fname << endl;
        count++;
    }

    //check if there's only one file with this runno
    if (count!=1)
    {
        cout << "ERROR - there's are " << count << " files. " << endl;
        return 2;
    }

    return Process(serv, fname, dummy);
}
