/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 08/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2006
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// filloptical.C
// =============
//
// This macro is used to read the files from KVA containing the results from
// the optical observations.
// Remark: Running it from the commandline looks like this:
//   root -q -l -b filloptical.C+\(\"filename\"\,kFALSE\) 
//
// Make sure, that database and password are corretly set.
//
// Returns 0 in case of failure and 1 in case of success.
//
///////////////////////////////////////////////////////////////////////////
#include <iostream>
#include <iomanip>
#include <fstream>

#include <TEnv.h>
#include <TRegexp.h>

#include <TSQLRow.h>
#include <TSQLResult.h>

#include "MDirIter.h"
#include "MSQLServer.h"
#include "MSQLMagic.h"

using namespace std;

//
// insert the entries from this optical data file into the db
//
int process(MSQLMagic &serv, TString fname)
{
    ifstream fin(fname);
    if (!fin)
    {
        cout << "Could not open file " << fname << endl;
        return 0;
    }

    TRegexp regexp("^20[0-9][0-9]-[0-1][0-9]-[0-3][0-9] [0-2][0-9]:[0-5][0-9]:[0-5][0-9].*$", kFALSE);

    Int_t numexp=0;
    Int_t numstars=0;

    TString timestamp;
    TString exposure;
    TString fitsfile;
    TString object;
    TString skylevel;
    TString fwhm;
    TString aperturer;
    TString mag;
    TString magerr;
    TString status;
    TString query;
    Int_t statuskey;
    Int_t objectkey;
    Int_t fitsfilekey;
    while (1)
    {
        TString line;
        line.ReadLine(fin);
        if (!fin)
            break;

        if (line=="KVA_Celestron_ST8  KVA_Celestron_R" || line.IsNull())
            continue;

        TObjArray *arr = line.Tokenize("  ");
        if (!line(regexp).IsNull())
        {
            if (arr->GetEntries()!=4)
            {
                cout << "WARNING: Timestampline with less or more than 4 arguments found " << endl;
                return 2;
            }
            numexp+=1;
            numstars=0;
            timestamp =Form("%s %s", (*arr)[0]->GetName(),(*arr)[1]->GetName());
            exposure   = (*arr)[2]->GetName();
            fitsfile = (*arr)[3]->GetName();
            continue;
        }
        else
        {
            if (arr->GetEntries()!=8)
            {
                cout << "WARNING: Objectline with less or more than 8 arguments found " << endl;
                return 2;
            }

            numstars+=1;
            object= Form("%s/%s", (*arr)[0]->GetName(),(*arr)[1]->GetName());
            skylevel   = (*arr)[2]->GetName();
            if (skylevel.Contains("-"))
                skylevel="NULL";
            fwhm = (*arr)[3]->GetName();
            if (fwhm=="-" || !fwhm.IsFloat())
                fwhm="NULL";
            aperturer = (*arr)[4]->GetName();
            mag = (*arr)[5]->GetName();
            if (!mag.IsFloat())
                mag="NULL";
            magerr = (*arr)[6]->GetName();
            if (!magerr.IsFloat())
                magerr="NULL";
            status = (*arr)[7]->GetName();
        }
        delete arr;

        if (numstars!=0)
        {
            /*
            cout << numexp << "th exposure: star # " << numstars << endl;
            cout << " timestamp: " << timestamp << endl;
            cout << " exposure : " << exposure  << endl;
            cout << " fitsfile : " << fitsfile  << endl;
            cout << " object   : " << object << endl;
            cout << " skylevel : " << skylevel  << endl;
            cout << " fwhm     : " << fwhm      << endl;
            cout << " aperturer: " << aperturer << endl;
            cout << " mag      : " << mag << " +/- " << magerr << endl;
            cout << " status   : " << status    << endl << endl;
            */
            statuskey = serv.QueryKeyOfName("Status", status.Data());
            objectkey = serv.QueryKeyOfName("Object", object.Data());
            fitsfilekey = serv.QueryKeyOfName("FitsFile", fitsfile.Data());
            query=Form("fTimeStamp='%s', fExposure=%s, fFitsFileKEY=%d, "
                       "fObjectKEY=%d, fSkyLevel=%s, fFWHM=%s, "
                       "fApertureRadius=%s, fInstrumentalMag=%s, "
                       "fInstrumentalMagErr=%s, fStatusKEY=%d ",
                       timestamp.Data(), exposure.Data(), fitsfilekey,
                       objectkey, skylevel.Data(), fwhm.Data(),
                       aperturer.Data(), mag.Data(), magerr.Data(),
                       statuskey);

            if (serv.Insert("OpticalData", query)==kFALSE)
                return 2;

        }

    }

    cout << fname(TRegexp("20[0-9][0-9]_[0-1][0-9]_[0-3][0-9]_KVA_C_R.*[.]instr", kFALSE))
        << ": " << setw(2) << numexp << " exposures, " <<  setw(2) << numstars << " stars" << endl;

    return 1;
}

// --------------------------------------------------------------------------
//
// loop over all files in this path
//
int filloptical(TString path, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLMagic serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return 0;
    }

    serv.SetIsDummy(dummy);

    cout << endl;
    cout << "filloptical" << endl;
    cout << "-----------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "Search Path: " << path << endl;
    cout << endl;

    //get all runbook files in path
    if (path.EndsWith(".instr"))
        return process(serv, path);

    //fill entries for each runbook file
    MDirIter Next(path, "2*_KVA_C_R*.instr", -1);
    while (1)
    {
        TString name = Next();
        if (name.IsNull())
            break;

        if (!process(serv, name))
            return 0;
    }

    return 1;
}
