/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 10/2006 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2004-2006
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////////
//
// starvisday.C
//
// displays the star visibility of several stars for one day
//
// See the code for ducumentation and setup
//
///////////////////////////////////////////////////////////////////////////

void starvisday()
{
    // Date (UTC) for which to get the visibility plot
    MTime time(-1); // -1 means NOW

    // Setup a different time as you like
    // time.Set(2006, 10, 10);

    // Current observatory (see class reference of MObservatory)
    const MObservatory obs(MObservatory::kMagic1);

    // The class contaning the catalog
    MAstroCatalog stars;
    stars.SetObservatory(obs);
    stars.SetTime(time);

    // Read the stars from a star catalog. The catalogs can be downloaded
    // from the Mars web page. For more information see class reference.
    stars.ReadXephem("/magic/datacenter/setup/magic_favorites.edb");

    // Mark the stars you would like to display (see the names in the catalog)
    stars.MarkObject("CrabNebula");
    stars.MarkObject("1ES1218+304");
    stars.MarkObject("1ES1426+428");
    stars.MarkObject("Mrk421");
    stars.MarkObject("Mrk501");
    stars.MarkObject("1ES1959+650");
    stars.MarkObject("1ES2344+514");

    // --------------------------------------------------------------------------
    //
    //  Start producing the nice plot
    //

    // open and setup a new canvas
    TCanvas *c = new TCanvas;
    c->SetBorderMode(0);
    c->SetFillColor(kWhite);
    c->SetFrameBorderMode(0);
    c->SetFrameFillStyle(0);

    c->SetGridx();
    c->SetGridy();

    // Draw the twilight and dark time
    TBox box;
    box.SetFillStyle(4100);
    for (int i=0; i<4; i++)
    {
        double set = obs.GetSunRiseSet(time.GetMjd()-1, -6*i)[1];
        double ris = obs.GetSunRiseSet(time.GetMjd(),   -6*i)[0];

        box.SetFillColor(19-i);
        box.DrawBox(MTime(set).GetAxisTime(),  5,
                    MTime(ris).GetAxisTime(), 90);
    }

    // Setup the TGraph which is drawn
    TGraph g;
    g.SetTitle(Form("Visibility at %s, %s",
                    obs.GetObservatoryName().Data(),
                    time.GetStringFmt("%A %e.%B %Y").Data()));
    g.SetLineWidth(2);

    // Some helper to iterate these three colors
    Int_t col[] = { kBlack, kBlue, kRed };

    // Loop over all stars in the catalog
    int k=0;
    TIter Next(stars.GetList());
    TObject *o=0;
    while ((o=Next()))
    {
        // Check if the star was previously marked
        if (!o->TestBit(1<<14))
            continue;

        // Set graph name to object name
        g.SetName(o->GetName());

        // produce visibility curve
        stars.GetVisibilityCurve(g);

        // Setup description
        TText txt;
        txt.SetTextFont(102);
        txt.SetTextSize(0.03);
        txt.SetTextColor(col[k%3]);

        // Setup color and style of TGraph
        g.SetLineColor(col[k%3]);
        g.SetLineStyle(1+k/3);

        // Draw TGraph
        g.DrawClone(k++?"c":"ac")->SetBit(kCanDelete);

        // Search for culmination
        Long64_t max = TMath::LocMax(g.GetN(), g.GetY());

        // draw name at culmination
        txt.DrawText(g.GetX()[max]-1.0*3600, g.GetY()[max]+1, o->GetName());
    }

    c->Modified();
    c->Update();

    // Draw a line for the "best observation conditions"
    TLine line;
    line.SetLineWidth(2);
    line.SetLineStyle(9);
    line.SetLineColor(13);

    TAxis &axe = *g.GetHistogram()->GetXaxis();
    line.DrawLine(axe.GetBinLowEdge(axe.GetFirst()), 65,
                  axe.GetBinLowEdge(axe.GetLast()),  65);

}
