/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2007
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibrateSignal
//
// This class is reading the output written by callisto. It calibrates
// signal and time.
//
// The signal and time extractors are read from the callisto-output. In
// pricipal you could overwrite these default using the resource file,
// but this is NOT recommended!
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibrateSignal.h"

#include <TEnv.h>
#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MDirIter.h"
#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

#include "MGeomCam.h"
#include "MHCamEvent.h"
#include "MPedestalCam.h"
#include "MBadPixelsCam.h"
#include "MArrivalTimeCam.h"

#include "MCalibrationQECam.h"
#include "MCalibrationBlindCam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationChargePINDiode.h"
#include "MCalibrationPatternDecode.h"

#include "MCalibrationChargeCalc.h"
#include "MCalibrationRelTimeCalc.h"

#include "MCalibConstCam.h"

#include "MHCalibrationChargeCam.h"
#include "MHCalibrationChargeBlindCam.h"
#include "MHCalibrationChargePINDiode.h"
#include "MHCalibrationRelTimeCam.h"
#include "MHCalibrationPulseTimeCam.h"
#include "MHVsTime.h"
#include "MHCamera.h"

#include "MCalibCalcFromPast.h"

#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MContinue.h"
#include "MTriggerPatternDecode.h"
#include "MFTriggerPattern.h"
#include "MFDataPhrase.h"
#include "MFilterList.h"
#include "MGeomApply.h"
#include "MPedestalSubtract.h"
#include "MPointingPosCalc.h"
#include "MPedCalcFromLoGain.h"
#include "MPedestalSubtract.h"
#include "MExtractor.h"
#include "MExtractTimeAndCharge.h"
#include "MExtractPINDiode.h"
#include "MExtractBlindPixel.h"
#include "MFCosmics.h"
#include "MTaskEnv.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsTreat.h"
#include "MFillH.h"
#include "MWriteRootFile.h"

#include "MFEvtNumber.h"
#include "MMoviePrepare.h"
#include "MMovieWrite.h"
#include "MImgCleanStd.h"

ClassImp(MJCalibrateSignal);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets:
// - fIsInterlaced to kTRUE
// - fIsRelTimesUpdate to kFALSE
// - fIsHiLoCalibration to kFALSE
//
MJCalibrateSignal::MJCalibrateSignal(const char *name, const char *title)
    : fExtractor(0), fIsInterlaced(kTRUE), fIsRelTimesUpdate(kTRUE), fIsMovieMode(kFALSE)
{
    fName  = name  ? name  : "MJCalibrateSignal";
    fTitle = title ? title : "Tool to calibrate data";
}

MJCalibrateSignal::~MJCalibrateSignal()
{
    if (fExtractor)
        delete fExtractor;
}

void MJCalibrateSignal::SetExtractor(const MExtractor *ext)
{
    if (fExtractor)
        delete fExtractor;

    fExtractor = ext ? (MExtractor*)ext->Clone() : NULL;
}

Bool_t MJCalibrateSignal::WriteResult() const
{
    if (IsNoStorage() || fIsMovieMode)
        return kTRUE;

    // FIXME: This is not nice because it will update the signal
    // file always. Which might make the usage outside of
    // callisto difficult.
    TObjArray cont;
    cont.Add(const_cast<TEnv*>(GetEnv()));

    if (fDisplay)
        cont.Add(fDisplay);

    const TString name(Form("signal%08d.root", fSequence.GetSequence()));
    return WriteContainer(cont, name, "UPDATE");
}

Bool_t MJCalibrateSignal::ReadCalibration(TObjArray &l, MBadPixelsCam &cam, MExtractor* &ext2, MExtractor* &ext3, TString &geom) const
{
    TString fname = Form("%s/calib%08d.root", fPathIn.Data(), fSequence.GetSequence());

    *fLog << inf << "Reading from file: " << fname << endl;

    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        *fLog << err << dbginf << "ERROR - Could not open file " << fname << endl;
        return kFALSE;
    }

    TObject *o = file.Get("ExtractSignal");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractSignal read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext3 = o ? (MExtractor*)o->Clone() : NULL;

    o = file.Get("ExtractTime");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractTime read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext2 = o ? (MExtractor*)o->Clone() : NULL;
    if (!ext3 && !ext2)
    {
        *fLog << err << dbginf << "ERROR - Neither ExtractSignal nor ExrtractTime found in " << fname << "!" << endl;
        return kFALSE;
    }

    o = file.Get("MGeomCam");
    if (o && !o->InheritsFrom(MGeomCam::Class()))
    {
        *fLog << err << dbginf << "ERROR - MGeomCam read from " << fname << " doesn't inherit from MGeomCam!" << endl;
        return kFALSE;
    }
    geom = o ? o->ClassName() : "";

    TObjArray cont(l);
    cont.Add(&cam);
    return ReadContainer(cont);
}

// --------------------------------------------------------------------------
//
// MJCalibration allows to setup several option by a resource file:
//   MJCalibrateSignal.RawData: yes,no
//
// For more details see the class description and the corresponding Getters
//
Bool_t MJCalibrateSignal::CheckEnvLocal()
{
    SetInterlaced(GetEnv("Interlaced", fIsInterlaced));
    SetRelTimesUpdate(GetEnv("RelTimesUpdate", fIsRelTimesUpdate));
    SetMovieMode(GetEnv("MovieMode", fIsMovieMode));

    return MJCalib::CheckEnvLocal();
}

Bool_t MJCalibrateSignal::Process(MPedestalCam &pedcamab, MPedestalCam &pedcambias,
                                  MPedestalCam &pedcamextr)
{
    if (!fSequence.IsValid())
    {
	  *fLog << err << "ERROR - Sequence invalid..." << endl;
	  return kFALSE;
    }

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate calibrated data from Sequence #";
    *fLog << fSequence.GetSequence() << endl << endl;


    //if (!CheckEnv())
    //    return kFALSE;

    CheckEnv();

    // --------------------------------------------------------------------------------

    MDirIter iter;
    if (fSequence.IsValid())
    {
        if (fSequence.SetupDatRuns(iter, 0, !fSequence.IsMonteCarlo())<=0)
            return kFALSE;
    }

    // Read File
    MHCalibrationChargeCam      hchacam;
    MHCalibrationChargeBlindCam hbndcam;
    MHCalibrationChargePINDiode hpndiod;
    MHCalibrationRelTimeCam     hrelcam;
    //MHCalibrationHiLoCam        hilocam;
    //MHCalibrationPulseTimeCam   hpulcam;

    hchacam.SetOscillations(kFALSE);
    hbndcam.SetOscillations(kFALSE);
    hrelcam.SetOscillations(kFALSE);

    MCalibrationChargeCam      calcam;
    MCalibrationQECam          qecam;
    MCalibrationBlindCam       bndcam;
    MCalibrationChargePINDiode pind;
    MCalibrationRelTimeCam     tmcam;
    //MCalibrationHiLoCam        hilcam;
    //MCalibrationPulseTimeCam   pulcam;

    MBadPixelsCam              badpix;

    TObjArray interlacedcont;
    if (fIsInterlaced)
    {
        interlacedcont.Add(&hchacam);
	if (IsUseBlindPixel())
            interlacedcont.Add(&hbndcam);
        if (IsUsePINDiode())  
            interlacedcont.Add(&hpndiod);
        if (fIsRelTimesUpdate)
            interlacedcont.Add(&hrelcam);
    }

    MExtractor *extractor1=fExtractor;
    MExtractor *extractor2=0;
    MExtractor *extractor3=0;
    TString geom;

    TObjArray calibcont;
    calibcont.Add(&calcam);
    calibcont.Add(&qecam);
    calibcont.Add(&bndcam);
    calibcont.Add(&tmcam);
    if (IsUseBlindPixel())
        calibcont.Add(&bndcam);

    if (!ReadCalibration(calibcont, badpix, extractor2, extractor3, geom))
        return kFALSE;

    *fLog << all;
    if (!geom.IsNull())
        *fLog << inf << "Camera geometry found in file: " << geom << endl;
    else
        *fLog << inf << "No Camera geometry found using default <MGeomCamMagic>" << endl;

    if (extractor3)
    {
        *fLog << underline << "Signal Extractor found in calibration file" << endl;
        extractor3->Print();
        *fLog << endl;
    }
    else
        *fLog << inf << "No Signal Extractor: ExtractSignal in file." << endl;


    if (extractor1)
    {
      *fLog << underline << "Modified Signal Extractor set by user." << endl;
      extractor1->Print();
      *fLog << endl;
    }
    else
    {
        *fLog << inf << "No modified Signal Extractor set by user... using ExtractSignal." << endl;
        extractor1 = extractor3 ? (MExtractor*)extractor3->Clone() : 0;
    }

    if (extractor2)
    {
        *fLog << underline << "Time Extractor found in calibration file" << endl;
        extractor2->Print();
        *fLog << endl;
    }
    else
        *fLog << inf << "No Time Extractor: ExtractTime in file." << endl;

    // This is necessary for the case in which it is not in the files
    MCalibConstCam constcam;

    //MBadPixelsCam badcam;
    constcam.SetBadPixels(&badpix);

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!
    plist.AddToList(&badpix);
    plist.AddToList(&constcam);
    //plist.AddToList(&hilcam);
    plist.AddToList(&calibcont);       // Using AddToList(TObjArray *)
    plist.AddToList(&interlacedcont);  // Using AddToList(TObjArray *)
    //plist.AddToList(&pulcam);

    // Setup Tasklist
    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadMarsFile readmc("Events");
    readmc.DisableAutoScheme();

    MRawFileRead rawread(NULL);

    MRead *read = fSequence.IsMonteCarlo() ? (MRead*)&readmc : (MRead*)&rawread;
    read->AddFiles(iter);

    const TString fname(Form("%s{s/_D_/_Y_}{s/\\.raw$/.root}{s/\\.raw\\.gz$/.root}", fPathOut.Data()));

    // Skips MC which have no contents. This are precisely the
    // events which fullfilled the MC Lvl1 trigger and an
    // arbitrary cut (typically at 50phe) to speed up simulation
    MContinue contmc("MRawEvtData.GetNumPixels<0.5", "ContEmptyMC");

    //MPointingPosInterpolate pextr;
    //pextr.AddFiles(&iter);
    
    MGeomApply             apply; // Only necessary to create geometry
    if (!geom.IsNull())
        apply.SetGeometry(geom);
    //MBadPixelsMerge        merge(&badpix);

    // Make sure that pedcamab has the correct name
    pedcamab.SetName("MPedestalFundamental");
    pedcamextr.SetName("MPedestalFromExtractorRndm");
    pedcambias.SetName("MPedestalFromExtractor");
    plist.AddToList(&pedcamextr);
    plist.AddToList(&pedcambias);
    plist.AddToList(&pedcamab);

    MArrivalTimeCam timecam;
    plist.AddToList(&timecam);

    // Check for interleaved events
    MCalibrationPatternDecode caldec;
    MTriggerPatternDecode     decode;

    // This will make that for data with version less than 5, where
    // trigger patterns were not yet correct, all the events in the real
    // data file will be processed. In any case there are no interleaved
    // calibration events in such data, so this is fine.
    MFTriggerPattern ftp;
    ftp.SetDefault(kTRUE);
    //    ftp.RequireCalibration();
    ftp.DenyCalibration();
    ftp.DenyPedestal();
    //    ftp.DenyPinDiode();
    ftp.SetInverted();

    // This will skip interleaved calibration events and pedestal events (if any)
    // --> tlist2
    MContinue conttp(&ftp, "ContTrigPattern");

    // Create the pedestal subtracted raw-data
    MPedestalSubtract pedsub;
    pedsub.SetPedestalCam(&pedcamab);

    // Do signal and pedestal calculation
    MPedCalcFromLoGain     pedlo1("MPedCalcFundamental");
    pedlo1.SetPedestalUpdate(kTRUE);
    pedlo1.SetNamePedestalCamOut("MPedestalFundamental");

    MPedCalcFromLoGain     pedlo2("MPedCalcWithExtractorRndm");
    pedlo2.SetPedestalUpdate(kTRUE);
    pedlo2.SetRandomCalculation(kTRUE);
    pedlo2.SetNamePedestalCamOut("MPedestalFromExtractorRndm");

    MPedCalcFromLoGain     pedlo3("MPedCalcWithExtractor");
    pedlo3.SetPedestalUpdate(kTRUE);
    pedlo3.SetRandomCalculation(kFALSE);
    pedlo3.SetNamePedestalCamOut("MPedestalFromExtractor");

    if (!extractor1)
    {
        *fLog << err << "ERROR - extractor1 == NULL" << endl;
        return kFALSE;
    }

    // Setup to use the hi-gain extraction window in the lo-gain
    // range (the start of the lo-gain range is added automatically
    // by MPedCalcFromLoGain)
    //
    // The window size of the extractor is not yet initialized,
    // so we have to stick to the extraction range
    //
    // Even if we would like to use a range comparable to the
    // hi-gain extraction we use the lo-gain range to make
    // sure that exclusions (eg. due to switching noise)
    // are correctly handled.
    //
    pedlo1.SetRangeFromExtractor(*extractor1);

    if (extractor1->InheritsFrom("MExtractTimeAndCharge"))
    {
        pedlo2.SetExtractor((MExtractTimeAndCharge*)extractor1);
        pedlo3.SetExtractor((MExtractTimeAndCharge*)extractor1);
    }
    else
    {
        pedlo2.SetRangeFromExtractor(*extractor1);
        pedlo3.SetRangeFromExtractor(*extractor1);
    }

    //------------------------------
    MFTriggerPattern ftp2;
    ftp2.SetDefault(kTRUE);
    ftp2.DenyCalibration();
    if (!extractor1->HasLoGain())
        ftp2.RequirePedestal();

    pedlo1.SetFilter(&ftp2);
    pedlo2.SetFilter(&ftp2);
    pedlo3.SetFilter(&ftp2);

    MContinue contftp2(&ftp2, "ContPedestal");
    //------------------------------

    MFCosmics fcosmics;
    fcosmics.SetNamePedestalCam("MPedestalFundamental");
    MContinue contcos(&fcosmics, "ContTrigEvts");
    contcos.SetInverted();

    //MMcPedestalCopy pcopy;
    MTaskEnv taskenv1("ExtractSignal");
    MTaskEnv taskenv2("ExtractTime");
    MTaskEnv taskenv3("ExtractInterlaced");
    taskenv1.SetDefault(extractor1);
    taskenv2.SetDefault(extractor2);
    taskenv3.SetDefault(extractor3);

    MFilterList flistftp2("PedestalFilter");
    flistftp2.SetInverted();
    flistftp2.AddToList(&ftp2);

    if (!extractor1->HasLoGain())
    {
        taskenv1.SetFilter(&flistftp2);
        taskenv2.SetFilter(&flistftp2);
        taskenv3.SetFilter(&flistftp2);
    }

    //
    // This is new calibration to photo-electrons, hard-coded
    // as decided at the Wuerzburg software meeting 26.01.05
    //
    MCalibrateData calib;
    calib.SetSignalType(MCalibrateData::kPhe);
    //calib.AddPedestal("Fundamental");
    calib.AddPedestal("FromExtractor");
    calib.AddPedestal("FromExtractorRndm");
    calib.SetPedestalFlag(MCalibrateData::kEvent);
    //----------------------------------------------------------

    MExtractPINDiode        pinext;
    MExtractBlindPixel      bldext;

    MFTriggerPattern        fcalib("CalibFilter");
    fcalib.SetDefault(kFALSE);
    fcalib.RequireCalibration();

    MCalibrationChargeCalc  chcalc;
    chcalc.SetContinousCalibration();
    chcalc.SetExtractor(extractor3);

    MCalibrationRelTimeCalc recalc;
    MCalibCalcFromPast      pacalc;

    chcalc.SetPedestals(&pedcamextr);

    pacalc.SetChargeCalc(&chcalc);
    if (fIsRelTimesUpdate)
      pacalc.SetRelTimeCalc(&recalc);
    pacalc.SetCalibrate(&calib);

    //
    // Calibration histogramming
    //
    MFillH filpin(&hpndiod, "MExtractedSignalPINDiode",   "FillPINDiode");
    MFillH filbnd(&hbndcam, "MExtractedSignalBlindPixel", "FillBlindCam");
    MFillH filcam(&hchacam, "MExtractedSignalCam",        "FillChargeCam");
    MFillH filtme(&hrelcam, "MArrivalTimeCam",            "FillRelTime");
    //MFillH filhil(&hilocam, "MExtractedSignalCam",        "FillHiLoRatio");
    //MFillH filpul(&hpulcam, "MRawEvtData",                "FillPulseTime");
    filpin.SetBit(MFillH::kDoNotDisplay);
    filbnd.SetBit(MFillH::kDoNotDisplay);
    filcam.SetBit(MFillH::kDoNotDisplay);
    filtme.SetBit(MFillH::kDoNotDisplay);
    //filhil.SetBit(MFillH::kDoNotDisplay);
    //filpul.SetBit(MFillH::kDoNotDisplay);

    MCalibrateRelTimes caltm;
    MBadPixelsCalc     bpcal;
    MBadPixelsTreat    treat;

    //bpcal.SetNamePedPhotCam("MPedPhotFromExtractor");
    bpcal.SetNamePedPhotCam("MPedPhotFromExtractorRndm");

    //treat.AddNamePedPhotCam("MPedPhotFundamental");
    treat.AddNamePedPhotCam("MPedPhotFromExtractor");
    treat.AddNamePedPhotCam("MPedPhotFromExtractorRndm");
    if (!extractor2 && !extractor1->InheritsFrom("MExtractTimeAndCharge"))
        treat.SetProcessTimes(kFALSE);

    MHCamEvent evt0(  0, "PedFLG",     "Fundamental Pedestal from Lo Gain;;P [cnts/sl]");
    MHCamEvent evt1(  2, "PedRmsFLG",  "RMS from Extractor applied to ped.;;\\sigma_{p} [cnts/sl]");
    MHCamEvent evt2(  0, "Extra'd",    "Extracted Signal;;S [cnts/sl]");
    // MHCamEvent evt3(4, "PedPhot",    "Calibrated Pedestal;;P [phe]");
    MHCamEvent evt4(  5, "PedRMS",     "Calibrated RMS from Extractor applied to ped.;;\\sigma_{p} [phe]");
    MHCamEvent evt5(  0, "Interp'd",   "Interpolated Signal scaled with A/A_{0};;S [phe]");
    MHCamEvent evt6(102, "Unsuitable", "Fraction of unsuitable events per Pixel;;[1]");
    // MHCamEvent evt7(  6, "Times",      "Calibrated Arrival Time;;T [fadc sl]");
    MHCamEvent evt8(  0, "Conv",       "Calibration Conv. Factors;;[phe/cnts]");
    MHCamEvent evt9(  7, "PulsePos",   "Pulse Position of cosmics (>50phe);;T");
    MHCamEvent evtR(  4, "HiLoCal",    "Hi-/Lo-Gain ratio;;Ratio");
    MHCamEvent evtO(  7, "HiLoOff",    "Lo-/Hi-Gain Offset;;Offset");
    evt2.SetErrorSpread(kFALSE);
    evt5.SetErrorSpread(kFALSE);
    evt6.SetErrorSpread(kFALSE);
    evt6.SetThreshold();

    MFillH fill0(&evt0, "MPedestalFundamental",          "FillPedFLG");
    MFillH fill1(&evt1, "MPedestalFromExtractorRndm",    "FillPedRmsFLG");
    MFillH fill2(&evt2, "MExtractedSignalCam",           "FillExtracted");
    // MFillH fill3(&evt3, "MPedPhotFundamental",  "FillPedPhot");
    MFillH fill4(&evt4, "MPedPhotFromExtractorRndm",     "FillPedRMS");
    MFillH fill5(&evt5, "MSignalCam",                    "FillInterpolated");
    MFillH fill6(&evt6, "MBadPixelsCam",                 "FillUnsuitable");
    // MFillH fill7(&evt7, "MSignalCam",                    "FillTimes");
    MFillH fill8(&evt8, "MCalibConstCam",                "FillConv");
    MFillH fill9(&evt9, "MSignalCam",                    "FillPulse");
    MFillH fillR(&evtR, "MExtractedSignalCam",           "FillHiLoCal");
    MFillH fillO(&evtO, "MArrivalTimeCam",               "FillHiLoOff");

    MHVsTime histbp("MBadPixelsCam.GetNumUnsuitable");
    histbp.SetName("BadPixTm");
    histbp.SetTitle("Number of unsuitable pixels;;N");
    histbp.SetMinimum(0);

    MHVsTime histdp("MSignalCam.GetNumPixelsUnmapped");
    histdp.SetName("DeadPixTm");
    histdp.SetTitle("Number of dead/unmapped pixels;;N");
    histdp.SetMinimum(0);

    // Task to fill the histogram
    MFillH fillB(&histbp, "MTime", "FillBadPixTm");
    MFillH fillD(&histdp, "MTime", "FillDeadPixTm");
    fillB.SetNameTab("BadPixTm");
    fillD.SetNameTab("DeadPixTm");

    /*
     MFillH fillP("MHPulseShape", "", "FillPulseShape");
     fillP.SetNameTab("Pulse");
     */

    if (!extractor1->HasLoGain())
    {
        fill0.SetFilter(&ftp2);
        fill1.SetFilter(&ftp2);
    }

    /*
     MHVsTime hbadpix("MBadPixelsCam.GetNumUnsuitable");
     hbadpix.SetNumEvents(50);
     MFillH fillB(&hbadpix, "MTime");
     */

    MTaskEnv fillflorian("FinalFantasy");
    fillflorian.SetDefault();

    // The second rule is for the case reading raw-files!
    MWriteRootFile write(2, fname, fOverwrite?"RECREATE":"NEW");
    // Run Header
    write.AddContainer("MRawRunHeader",             "RunHeaders");
//    write.AddContainer("MBadPixelsCam",             "RunHeaders");
    write.AddContainer("MGeomCam",                  "RunHeaders");
    // Monte Carlo Headers
    write.AddContainer("MMcRunHeader",              "RunHeaders", kFALSE);
    write.AddContainer("MMcFadcHeader",             "RunHeaders", kFALSE);
    write.AddContainer("MMcTrigHeader",             "RunHeaders", kFALSE);
    write.AddContainer("MMcConfigRunHeader",        "RunHeaders", kFALSE);
    write.AddContainer("MMcCorsikaRunHeader",       "RunHeaders", kFALSE);
    // Monte Carlo
    write.AddContainer("MMcEvt",                    "Events",     kFALSE);
    write.AddContainer("MMcTrig",                   "Events",     kFALSE);
    // Data tree
    write.AddContainer("MSignalCam",                "Events");
    // write.AddContainer("MPedPhotFundamental",               "Events");
    write.AddContainer("MPedPhotFromExtractor",     "Events");
    write.AddContainer("MPedPhotFromExtractorRndm", "Events");
    write.AddContainer("MTime",                     "Events",     kFALSE);
    write.AddContainer("MRawEvtHeader",             "Events");

    // Slow-Control: Current-tree
    write.AddContainer("MTimeCurrents",             "Currents",   kFALSE);
    write.AddContainer("MCameraDC",                 "Currents",   kFALSE);
    write.AddContainer("MReportCurrents",           "Currents",   kFALSE);
    // Slow-Control: Camera-tree
    write.AddContainer("MReportCamera",             "Camera",     kFALSE);
    write.AddContainer("MTimeCamera",               "Camera",     kFALSE);
    write.AddContainer("MCameraAUX",                "Camera",     kFALSE);
    write.AddContainer("MCameraCalibration",        "Camera",     kFALSE);
    write.AddContainer("MCameraCooling",            "Camera",     kFALSE);
    write.AddContainer("MCameraHV",                 "Camera",     kFALSE);
    write.AddContainer("MCameraLV",                 "Camera",     kFALSE);
    write.AddContainer("MCameraLids",               "Camera",     kFALSE);
    // Slow-Control: Trigger-tree
    write.AddContainer("MReportTrigger",            "Trigger",    kFALSE);
    write.AddContainer("MTimeTrigger",              "Trigger",    kFALSE);
    // Slow-Control: Drive-tree
    write.AddContainer("MReportDrive",              "Drive",      kFALSE);
    write.AddContainer("MTimeDrive",                "Drive",      kFALSE);
    // Slow-Control: Central Control-tree
    write.AddContainer("MReportCC",                 "CC",         kFALSE);
    write.AddContainer("MTimeCC",                   "CC",         kFALSE);

    // Write the special MC tree
    MWriteRootFile writemc(2, fname, fOverwrite?"RECREATE":"NEW");
    writemc.SetName("WriteMC");
    writemc.AddContainer("MMcEvtBasic", "OriginalMC");

    // Write the special calib tree
    /*
    MWriteRootFile writecal(2, fname, fOverwrite?"RECREATE":"NEW");
    writecal.SetName("WriteCalib");
    writecal.AddContainer("MBadPixelsCam",          "Calib");
    writecal.AddContainer("MCalibrationChargeCam",  "Calib");
    writecal.AddContainer("MCalibrationRelTimeCam", "Calib");
    */

    // Now setup tasklist for events
    MTaskList tlist2;

    tlist2.AddToList(&caldec);
    tlist2.AddToList(&decode);
    tlist2.AddToList(&apply);
    //tlist2.AddToList(&merge);
    tlist2.AddToList(&pedsub);
    tlist2.AddToList(&ftp2);
    tlist2.AddToList(&pedlo1);
    tlist2.AddToList(&pedlo2);
    tlist2.AddToList(&pedlo3);
    tlist2.AddToList(&fill0);        // fill pedestal events
    tlist2.AddToList(&fill1);        // fill pedestal events

    //-----------------------------------------------------------

    MTaskList tlist3;
    tlist3.SetFilter(&fcalib);

    //MFDataPhrase filcalco("MCalibrationConstCam.IsReadyToSave>0.5", "CalibConstFilter");
    if (fIsInterlaced)
    {
        tlist2.AddToList(&fcalib);     // MFTriggerPattern
        tlist2.AddToList(&tlist3);
        if (IsUsePINDiode())
            tlist3.AddToList(&pinext); // MExtractPINDiode
        if (IsUseBlindPixel())
            tlist3.AddToList(&bldext); // MExtractBlindPixel
	tlist3.AddToList(&taskenv3);
        tlist3.AddToList(&pacalc);     // MCalibCalcFromPast
        /*
         tlist3.AddToList(&filcalco);   // CalibConstFilter (IsReadyToSave)
         fill8.SetFilter(&filcalco);
         tlist3.AddToList(&fill8);      // FillConvUpd
         */

        tlist3.AddToList(&filcam);     // FillChargeCam
	if (fIsRelTimesUpdate)
	  tlist3.AddToList(&filtme);   // FillRelTime
        if (IsUseBlindPixel())
            tlist3.AddToList(&filbnd); // FillBlindCam
        if (IsUsePINDiode())
            tlist3.AddToList(&filpin); // FillPINDiode
        tlist3.AddToList(&chcalc);     // MCalibrationChargeCalc
	if (fIsRelTimesUpdate)
            tlist3.AddToList(&recalc); // MCalibrationRelTimeCam

        //tlist3.AddToList(&writecal);   // MWriteRootFile
    }

    // Continue for all non-cosmic events
    if (!extractor1->HasLoGain())
        tlist2.AddToList(&contftp2); // remove pedestal events from processing
    tlist2.AddToList(&conttp);       // remove calib events from processing
    if (extractor1)
      tlist2.AddToList(&taskenv1);
    if (extractor2)
      tlist2.AddToList(&taskenv2);
    tlist2.AddToList(&contcos);      // MFCosmics (ContTrigEvts)
    /*
    if (fIsHiLoCalibration)
    {
        plist.AddToList(&hilocam);
        tlist2.AddToList(&filhil);
        }

    if (fIsPulsePosCheck)
    {
        plist.AddToList(&hpulcam);
        tlist2.AddToList(&filpul);
    }
    */
    tlist2.AddToList(&fill2);
    tlist2.AddToList(&fill8);        // FillConv
    tlist2.AddToList(&calib);        // MCalibrateData
    if (extractor2 || extractor1->InheritsFrom("MExtractTimeAndCharge"))
        tlist2.AddToList(&caltm);

    tlist2.AddToList(&bpcal);        // MBadPixelsCalc
    tlist2.AddToList(&treat);        // MBadPixelsTreat
    tlist2.AddToList(&fill6);
    //    tlist2.AddToList(&fill3);
    tlist2.AddToList(&fill4);
    tlist2.AddToList(&fill5);
    //if (extractor2 || extractor1->InheritsFrom("MExtractTimeAndCharge"))
    //    tlist2.AddToList(&fill7);
    tlist2.AddToList(&fill9);
    if (!fSequence.IsMonteCarlo())
    {
        tlist2.AddToList(&fillB);
        tlist2.AddToList(&fillD);
    }
    if (extractor1->HasLoGain())
    {
        tlist2.AddToList(&fillR);
        tlist2.AddToList(&fillO);
    }

    /*
     MFillH fillC("MHCleaning", "", "FillClean");
     tlist2.AddToList(&fillC);

     //tlist2.AddToList(&fillP);
     */

    // ----- Start: Code for encoding movies -----

    MMoviePrepare movprep;
    MMovieWrite   movwrite;
    movprep.SetRangeFromExtractor(*extractor1);

    //MFDataPhrase movfilt("MMovieData.fMax>150");
    MFDataPhrase movfilt("MMovieData.fMax>5*MMovieData.fMedianPedestalRms", "MovieFilter");

    MImgCleanStd movclean(8.5, 4.0);
    movclean.SetMethod(MImgCleanStd::kAbsolute);

    //movprep.SetFilter(&evtnum);
    movclean.SetFilter(&movfilt);
    movwrite.SetFilter(&movfilt);

    MTaskList tlistmov("MovieEncoder");
    tlistmov.AddToList(&movprep);
    tlistmov.AddToList(&movfilt);
    tlistmov.AddToList(&movclean);
    tlistmov.AddToList(&movwrite);

    MFEvtNumber evtnum;
    //evtnum.SetSelector("ThetaSquared.fVal<0.04");
    //evtnum.SetFileName("ganymed00000001.root");
    tlistmov.SetFilter(&evtnum);

    if (fIsMovieMode)
    {
        tlist2.AddToList(&evtnum);
        tlist2.AddToList(&tlistmov);
    }

    // ----- End: Code for encoding movies -----

    tlist2.AddToList(&fillflorian);

    // Setup List for Drive-tree
    //MPointingPosCalc pcalc;

    // Now setup main tasklist
    tlist.AddToList(read);

    if (fSequence.IsMonteCarlo())
    {
        if (!fIsMovieMode)
            tlist.AddToList(&writemc);
        tlist.AddToList(&contmc);
    }

    //if (IsUseRootData())
    //  tlist2.AddToList(&pextr);
    tlist.AddToList(&tlist2, fSequence.IsMonteCarlo() ? "Events" : "All");

    //if (fSequence.IsMonteCarlo())
    //  tlist.AddToList(&pcalc, "Drive");

    if (!fIsMovieMode)
        tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    // Execute first analysis
    const Bool_t rc = evtloop.Eventloop(fMaxEvents);

    // make sure owned object are deleted
    if (extractor1 && extractor1!=fExtractor)
        delete extractor1;
    if (extractor2)
        delete extractor2;
    if (extractor3)
        delete extractor3;

    // return if job failed
    if (!rc)
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    // if everything went ok write and display result
    DisplayResult(plist);

    /*
    if (fIsPixelCheck)
      {
        if (fIsPulsePosCheck)
            hpulcam[fCheckedPixId].DrawClone("");

	//if (fIsHiLoCalibration)
	//    hilocam[fCheckedPixId].DrawClone("");
      }
    interlacedcont.Add(&pulcam);
    */

    //if (fIsHiLoCalibration)
    //  interlacedcont.Add(&hilcam);

    //if (fIsPulsePosCheck)
    //    interlacedcont.Add(plist.FindObject("MHCalibrationPulseTimeCam"));

    //if (fIsHiLoCalibration)
    //    interlacedcont.Add(plist.FindObject("MHCalibrationHiLoCam"));

    if (!WriteResult())
        return kFALSE;

    // return if job went ok
    *fLog << all << GetDescriptor() << ": Done." << endl;
    *fLog << endl << endl;

    return kTRUE;
}


void MJCalibrateSignal::DisplayResult(MParList &plist)
{
    /*
    if (!fDisplay || !fIsHiLoCalibration)
        return;

    MCalibrationHiLoCam *hcam = (MCalibrationHiLoCam*)plist.FindObject("MCalibrationHiLoCam");
    MGeomCam            *geom = (MGeomCam*)plist.FindObject("MGeomCam");
    if (!hcam || !geom)
        return;

    // Create histograms to display
    MHCamera disp1(*geom, "HiLoConv", "Ratio Amplification HiGain vs. LoGain (Charges)");
    MHCamera disp2(*geom, "HiLoDiff", "Arrival Time Diff. HiGain vs. LoGain (Times)");

    disp1.SetCamContent(*hcam, 0);
    disp1.SetCamError(  *hcam, 1);
    disp2.SetCamContent(*hcam, 5);
    disp2.SetCamError(  *hcam, 6);

    disp1.SetYTitle("R [1]");
    disp2.SetYTitle("\\Delta T [FADC sl.]");

    TCanvas &c1 = fDisplay->AddTab("HiLoConv");
    c1.Divide(2,3);

    disp1.CamDraw(c1, 1, 2, 1);
    disp2.CamDraw(c1, 2, 2, 1);
*/
}

