/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MDataCheck.h"

#include <TGButton.h>  // TGTextButton

// ---

#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"
#include "MReadTree.h"
#include "MFillH.h"
#include "MGDisplayAdc.h"

// ---

ClassImp(MDataCheck)

enum {
  M_BUTTON_PEDADC,
  M_BUTTON_CRADC,
  M_BUTTON_PEDTDC,
  M_BUTTON_CRTDC
};

// --------------------------------------------------------------------------
//
//  Create the 'Data Check' GUI Controls (Window) on the screen. To use it
//  from within the interpreter you can call a Standard Version with
//  'new MDataCheck()'
//
MDataCheck::MDataCheck(const TGWindow *main, const TGWindow *p,
                       const UInt_t w, const UInt_t h)
: MBrowser(main, p, w, h)
{
    TGTextButton *pedadc = new TGTextButton(fTop2, "ADC Spectra of Pedestals", M_BUTTON_PEDADC);
    TGTextButton *cradc  = new TGTextButton(fTop2, "ADC Specta of Cosmics",    M_BUTTON_CRADC);
    TGTextButton *pedtdc = new TGTextButton(fTop3, "TDC Spectra of Pedestals", M_BUTTON_PEDTDC);
    TGTextButton *crtdc  = new TGTextButton(fTop3, "TDC Specta of Cosmics",    M_BUTTON_CRTDC);

    pedadc->Associate(this);
    cradc ->Associate(this);
    pedtdc->Associate(this);
    crtdc ->Associate(this);

    fList->Add(pedadc);
    fList->Add(cradc);
    fList->Add(pedtdc);
    fList->Add(crtdc);

    TGLayoutHints *laybut = new TGLayoutHints(kLHintsTop|kLHintsLeft, 10, 10, 5, 5);
    fList->Add(laybut);

    fTop2->AddFrame(pedadc, laybut);
    fTop2->AddFrame(cradc,  laybut);

    fTop3->AddFrame(pedtdc, laybut);
    fTop3->AddFrame(crtdc,  laybut);

    MapSubwindows();

    Layout();

    SetWindowName("DataCheck Window");
    SetIconName("DataCheck");

    MapWindow();
}

// --------------------------------------------------------------------------
//
//  Create the 'View Adc' GUI Controls (Window) on the screen.
//  Therefor we have to process all data in a file and fill the corresponding
//  histograms.
//
void MDataCheck::ViewAdcSpectra(Char_t *inputfile, Char_t *treeName)
{
    //
    // create a (empty) list of parameters which can be used by the tasks
    // and an (empty) list of tasks which should be executed
    // connect them in the required way.
    //

    //
    //   create the data containers for the raw data
    //
    MParList plist;

    //
    // set up the tasks for this job
    //
    MTaskList tasks;
    plist.AddToList(&tasks);

    MReadTree readin(treeName, inputfile);
    tasks.AddToList(&readin);

    MFillH fillspect("MRawEvtData", "MHFadcCam");
    tasks.AddToList(&fillspect);

    //
    // set up the loop for the processing
    //
    MEvtLoop magic;
    magic.SetParList(&plist);

    //
    // start the loop running
    //
    if (!magic.Eventloop())
        return;

    new MGDisplayAdc((MHFadcCam*)plist.FindObject("MHFadcCam"));
}

// --------------------------------------------------------------------------
//
//  Process the GUI control events (here: mainly button clicks)
//
Bool_t MDataCheck::ProcessMessage(Long_t msg, Long_t parm1, Long_t parm2)
{
    // Process events generated by the buttons in the frame.

    if (GET_MSG(msg)!=kC_COMMAND || GET_SUBMSG(msg)!=kCM_BUTTON)
        return MBrowser::ProcessMessage(msg, parm1, parm2);

    switch (parm1)
    {
    case M_BUTTON_PEDADC:
    case M_BUTTON_CRADC:
    case M_BUTTON_PEDTDC:
    case M_BUTTON_CRTDC:
        if (!InputFileSelected())
        {
            DisplError("No Input (root) File selected!");
            return kTRUE;
        }

        switch (parm1)
        {
        case M_BUTTON_PEDADC:
            ViewAdcSpectra(fInputFile, "PedEvents");
            return kTRUE;

        case M_BUTTON_CRADC:
            ViewAdcSpectra(fInputFile, "Events");
            return kTRUE;

        case M_BUTTON_PEDTDC:
            // fOctober.PedTdcSpectra(fInputFile) ;
            return kTRUE;

        case M_BUTTON_CRTDC:
            return kTRUE;
        }
        return kTRUE;
    }

    return MBrowser::ProcessMessage(msg, parm1, parm2);
}
