//=//////////////////////////////////////////////////////////////////////
//=
//= creadparam                
//=
//= @file        creadparam.h
//= @desc        Header file
//= @author      J C Gonzalez
//= @email       gonzalez@mppmu.mpg.de
//= @date        Thu May  7 16:24:22 1998
//=
//=----------------------------------------------------------------------
//=
//= Created: Thu May  7 16:24:22 1998
//= Author:  Jose Carlos Gonzalez
//= Purpose: Program for reflector simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: creadparam.h,v $
//= $Revision: 1.3 $
//= $Author: blanch $ 
//= $Date: 2000-03-24 18:15:46 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |creadparam.h|.

/*!@"

  In this section you can find the source code for the file
  |creadparam.h|.  This file is mainly needed by
  |creadparam.cxx|. Here is located the definition of the commands you
  can use in the parameters file. In this file, the first line must be
  |camera 'version'|, where |'version'| is the appropiate version of
  the output format (NOT the version of the camera program) which can
  read the commands written in that parameters file. You cannot mix
  parameters files and executables with different versions. The
  current version is |camera 0.2|.

  The commands now available for the parameters file are:

  @itemize
  
  @- |input_file| filename :    
     Sets the name of the input file (|.rfl|).
  @- |starfield_file| filename :    
     Sets the name of the starfield input file (|.rfl|).
  @- |ct_file| filename :    
     Sets the name of the CT definition file (|.def|).
  @- |data_file| filename :    
     Sets the name of the output data file (|.dat|).
  @- |nsb_on| :    
     Activates the NSB simulation. This is the default.
  @- |nsb_off| :    
     De-activates the NSB simulation.
  @- |nsb_mean| number :    
     Sets the mean value for the NSB.
     Default value: 6 for CT1, 6 for MAGIC.
     This implies always |nsb_on|.
  @- |threshold| number :    
     Sets the Threshold value q0. Default value: 10.
  @- |tail_cut| number : 
     Sets the Tail-Cut value.
     Default value: 7.
  @- |islands_cut| number :    
     Sets the Islands-Cut value i0.
     Default value: 10.
  @- |end_file|
     Last command in the parameters file.

  @enditemize

  @ignoreHTML
  A parameters file (a small one) looks like this:

  |camera 0.2|

  |input_file    gm100-500.rfl|

  |output_file   gm100-500.phe|

  |output_file   gm100-500.dat|

  |ct_file       magic.def|

  |threshold     10.0|

  |tail_cut      7.0|

  |nsb_mean      5.0|

  |end_file|
  @endignoreHTML

@"*/

//!@{

#ifndef _creadparam_
#define _creadparam_

#ifndef _this_
#define _this_ creadparam
#endif

//!@}

//!@subsection Include files.

//!@{

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>

#include "jcmacros.h"
#include "jcdebug.h"

#include "camera-v.h"

//!@}

//!@subsection Macro-definitions, and constants.

//!@{

// now we define the list ITEM_LIST of possible items in
// the parameters file. note that they are functions of
// T(x). we will change T(x) to be the real item or the
// string of this item when needed

#define ITEM_LIST   /* LIST OF ITEMS IN THE PARAMETERS FILE */     \
T(input_file),      /* input file */                              \
T(starfield_file),  /* starfield file */                              \
T(data_file),       /* data file */                              \
T(diag_file),       /* diagnostic output file (ROOT format) */   \
T(root_file),       /* ROOT file */                              \
T(ct_file),         /* file with the characteristics of the CT */  \
T(ana_pixels),      /* size of the camera for parameters calculation */  \
T(nsb_on),          /* activates NSB simulation */ \
T(nsb_off),         /* de-activates NSB simulation */ \
T(nsb_mean),        /* mean value of NSB contribution per pixel */ \
T(threshold),       /* value of q0 for trigger */ \
T(tail_cut),        /* value of tail cut (t0) */ \
T(islands_on),      /* DO count islands */ \
T(islands_off),     /* do NOT count islands */ \
T(islands_cut),     /* value of islands cut (i0) */ \
T(seeds),           /* seeds for random number generation */ \
T(data_from_stdin), /* to read data from STDIN */ \
T(skip),            /* skip pathological showers */ \
T(read_phe_all),    /* id., but was processed with write_all_images */ \
T(read_phe),        /* read an already camera processed file */ \
T(write_all_images),/* write to file .phe ALL images (even w.o. trigger)*/ \
T(nowrite_McEvt),   /* nowrite to file .root McEvt infos */ \
T(write_McTrig),    /* write to file .root McTrig infos */ \
T(write_RawEvt),    /* write to file .root RawEvt infos */ \
T(write_all_data),  /* write to file .dat ALL image data */ \
T(select_energy),   /* energy range to read: only for .phe files */ \
T(trigger_radius),  /* trigger radius for the camera */ \
T(correction),      /* factor for correction in the pixel values */ \
T(trigger_scan),    /* show signal in the diskriminator */ \
T(fadc_scan),       /* show fadc signal */ \
T(end_file)         /* end of the parameters file */
  
#define T(x)  x             // define T() as the name as it is

enum ITEM_TYPE {
  ITEM_LIST
};

#undef T

#define T(x)  #x              // define T() as the string of x

const char *const ITEM_NAMES[] = {
  ITEM_LIST
};

#undef T

#define LINE_MAX_LENGTH  400
#define ITEM_MAX_LENGTH  40
#define PATH_MAX_LENGTH  120

// mean values of NSB contribution per pixel

static const float Mean_NSB_MAGIC = 5.0; //@< for MAGIC
static const float Mean_NSB_CT1 = 5.0;   //@< for CT1

//!@}

//!@subsection Prototypes of functions.

//!@{

//++
// prototypes
//--

void readparam(char * filename);
char *get_input_filename(void);
char *get_starfield_filename(void);
char *get_data_filename(void);
char *get_diag_filename(void);
char *get_root_filename(void);
char *get_ct_filename(void);
int get_nsb(float *n);
float get_threshold(void);
float get_tail_cut(void);
int get_islands_cut(int *n);
long int get_seeds(int n);
int get_ana_pixels(void);
void get_skip_showers( int *s ); 
int get_nskip_showers( void ); 
int get_data_from_stdin(void);
int get_read_phe(void);
int get_read_phe_all(void);
int get_write_all_images(void);
//
int get_write_McEvt(void);
int get_write_McTrig(void);
int get_write_RawEvt(void);
//
int get_write_all_data(void);
int get_select_energy(float *le, float *ue);
int get_trigger_radius(float *radius);
int get_correction(float *corr);
int get_FADC_Scan(void);
int get_Trigger_Scan(void);
//!@}

//!@{

#endif // ! _creadparam_

//!@}

//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{

/*
 * $Log: not supported by cvs2svn $
 * Revision 1.2  2000/02/18 17:50:07  petry
 * This version belongs to camera.cxx 1.5.
 * It version has been put in the repository in order to be
 * able to share the further development with others.
 *
 * If you need something working, wait or take an earlier one.
 * See file README
 *
 * Revision 1.1.1.1  1999/11/05 11:59:34  harald
 * This the starting point for CVS controlled further developments of the
 * camera program. The program was originally written by Jose Carlos. 
 * But here you can find a "rootified" version to the program. This means 
 * that there is no hbook stuff in it now. Also the output of the
 * program changed to the MagicRawDataFormat. 
 *
 * The "rootification" was done by Dirk Petry and Harald Kornmayer. 
 *
 * Revision 1.2  1999/10/22 15:01:29  petry
 * version sent to H.K. and N.M. on Fri Oct 22 1999
 *
 * Revision 1.1.1.1  1999/10/21 16:35:11  petry
 * first synthesised version
 *
 * Revision 1.7  1999/03/15  14:59:09  gonzalez
 * camera-1_1
 *
 * Revision 1.6  1999/03/02  09:56:13  gonzalez
 * *** empty log message ***
 *
 * Revision 1.5  1999/01/14  17:32:43  gonzalez
 * Added to camera the STDIN input option (data_from_input)
 *
 */

//!@}
//=EOF
