#ifndef __MTrigger__
#define __MTrigger__

//     class MTrigger
//
//     implemented by Harald Kornmayer
//
//     This is a class to simulate the trigger. 
//     It assumes a special response of the PMT for one single Photo-electron. 
//     
//
//
#include <iostream.h>
#include <math.h> 

#include "TROOT.h"
#include "TObject.h"
#include "TRandom.h"
#include "TH1.h"

#include "Mdefine.h" 
#include "MMcEvt.h"

#include "MTriggerDefine.h"

//==========
//  MTrigger
//
//  The simulation of the Trigger for MonteCarlo Events is using this 
//  class. So all methods concerning the trigger should be done inside this
//  class. 
//
//  For a better understanding of the behavior of the trigger is here small
//  abstract of the trigger. This may change in the future. 
//
//  
//  We now from the camera program (This is the surrounding of the class 
//  MTrigger.) that one photo electron leaves at time t the photo cathode 
//  of the pixel number iPix). 
//
//  At the end of the PMT, the preamp, the optical fiber transmission we
//  get a signal of a given shape. After some discussion with Eckart the 
//  standard response function looks like this :    
//                                  
//  It is a gaussian Signal with a given FWHM. 
//
//  So whenever a photo electron leaves the photo cathod, on has to add
//  the standard response function to the analog signal of the pixel. 
//
//  Each pixel of the camera has such an summed-up analog signal. It may 
//  look like this picture: 
//
//
//  This is the input of the discriminator for the pixels. The output of
//  the discriminator is a digital signal. The response of the diskriminator
//  is not fixed at the moment. There are discussion about this topic. 
//  
//  At the moment the response is very simple. Whenever the analog signal
//  is crossing a defined threshold from below to above, a digital signal 
//  with a given length is created. 
// 
//  Now one can start with the simulation of different trigger levels. 
//  
//  The TriggerLevelZero is a very easy one. It is just looking if there 
//  are more then N digital signals at level ON (=1). If this is the case,
//  a TriggerLevelZero signal is created.
//
//  The TriggerLevelOne is implemented now. This is be a kind of next 
//  neighbour condition (i.e. four neigbouring analog signals at the same 
//  time, but this requests at least four digital signals at level ON, what 
//  is equivalent with a TriggerLevelZero.   
//  
//  
class MTrigger {

 private:
  //
  //    then for all pixels the shape of all the analog signals 
  //
  Bool_t   used [TRIGGER_PIXELS] ;  //  a boolean to indicated if the pixels is used in this event
  Int_t    nphot[TRIGGER_PIXELS];   //  count the photo electrons per pixel (NSB phe are not counted) 
 
  Float_t  *a_sig[TRIGGER_PIXELS] ; //  the analog signal for pixels

  Float_t  baseline[TRIGGER_PIXELS] ; //  for the baseline shift

  //
  //    then for all pixels the shape of the digital signal
  //
  Bool_t  dknt [TRIGGER_PIXELS] ;  //  a boolean to indicated if the pixels has passed the diskrminator 
  Float_t *d_sig[TRIGGER_PIXELS] ; //  the digital signal for all pixels

  //
  //    and the sum of all digital signals
  // 
  Float_t sum_d_sig[TRIGGER_TIME_SLICES] ; 


  //
  //    first the data for the response function
  //
  Float_t fwhm_resp ;                      // fwhm of the phe_response function 
  Float_t ampl_resp ;                      // amplitude of the phe_response function (in mV)
  Float_t sing_resp[ RESPONSE_SLICES ] ;   // the shape of the phe_response function 

  TH1F     *histPmt ; 
  Float_t  histMean ;    // Mean value of the distribution of Rasmik (stored in histPmt) 
  TRandom  *GenElec  ;   // RandomGenerator for the Electronic Noise

  //
  //    some values for the trigger settings
  //
  
  Float_t chan_thres ; // the threshold (in mV) for each individuel pixels
  Float_t gate_leng  ; // the length of the digital signal if analog signal is above threshold

  Float_t trigger_multi  ;  // Number of Pixels requested for a Trigger

  //
  //  The lookup table for the next neighbours
  // 

  Int_t NN[TRIGGER_PIXELS][6] ; 
 
  //
  //    some information about the different TriggerLevels in each Event
  // 

  Int_t  nZero ;         // how many ZeroLevel Trigger in one Event
  Int_t  SlicesZero[5] ; // Times Slices at which the ZeroLevel Triggers occur

  Int_t  nFirst ;         // how many FirstLevel Trigger in one Event
  Int_t  SlicesFirst[5] ; // Times Slices at which the FirstLevel Triggers occur

  Int_t  nSecond ;         // how many SecondLevel Trigger in one Event
  Int_t  SlicesSecond[5] ; // Times Slices at which the SecondLevel Triggers occur


public:

  MTrigger() ; 

  ~MTrigger() ; 

  void Reset() ; 

  Float_t  Fill( Int_t, Float_t ) ;  

  Float_t  FillNSB( Int_t, Float_t ) ;  

  void ElecNoise() ;

  Int_t Diskriminate() ;

  Int_t FirstLevel() ;   

  Bool_t PassNextNeighbour( Bool_t m[] ) ; 
 
  void ShowSignal (MMcEvt *McEvt) ; 

  Float_t GetFirstLevelTime( Int_t il ) ; 

} ; 

#endif

