//=//////////////////////////////////////////////////////////////////////
//=
//= creadparam                
//=
//= @file        creadparam.h
//= @desc        Header file
//= @author      J C Gonzalez
//= @email       gonzalez@mppmu.mpg.de
//= @date        Thu May  7 16:24:22 1998
//=
//=----------------------------------------------------------------------
//=
//= Created: Thu May  7 16:24:22 1998
//= Author:  Jose Carlos Gonzalez
//= Purpose: Program for reflector simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: creadparam.h,v $
//= $Revision: 1.1.1.1 $
//= $Author: harald $ 
//= $Date: 2000-02-08 15:13:44 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |creadparam.h|.

/*!@"

  In this section you can find the source code for the file
  |creadparam.h|.  This file is mainly needed by
  |creadparam.cxx|. Here is located the definition of the commands you
  can use in the parameters file. In this file, the first line must be
  |camera 'version'|, where |'version'| is the appropiate version of
  the output format (NOT the version of the camera program) which can
  read the commands written in that parameters file. You cannot mix
  parameters files and executables with different versions. The
  current version is |camera 0.2|.

  The commands now available for the parameters file are:

  @itemize
  
  @- |input_file| filename :    
     Sets the name of the input file (|.rfl|).
  @- |output_file| filename :    
     Sets the name of the output file (|.phe|).
  @- |ct_file| filename :    
     Sets the name of the CT definition file (|.def|).
  @- |data_file| filename :    
     Sets the name of the output data file (|.dat|).
  @- |nsb_on| :    
     Activates the NSB simulation. This is the default.
  @- |nsb_off| :    
     De-activates the NSB simulation.
  @- |nsb_mean| number :    
     Sets the mean value for the NSB.
     Default value: 6 for CT1, 6 for MAGIC.
     This implies always |nsb_on|.
  @- |threshold| number :    
     Sets the Threshold value q0. Default value: 10.
  @- |tail_cut| number : 
     Sets the Tail-Cut value.
     Default value: 7.
  @- |islands_cut| number :    
     Sets the Islands-Cut value i0.
     Default value: 10.
  @- |end_file|
     Last command in the parameters file.

  @enditemize

  @ignoreHTML
  A parameters file (a small one) looks like this:

  |camera 0.2|

  |input_file    gm100-500.rfl|

  |output_file   gm100-500.phe|

  |output_file   gm100-500.dat|

  |ct_file       magic.def|

  |threshold     10.0|

  |tail_cut      7.0|

  |nsb_mean      5.0|

  |end_file|
  @endignoreHTML

@"*/

//!@{

#ifndef _creadparam_
#define _creadparam_

#ifndef _this_
#define _this_ creadparam
#endif

//!@}

//!@subsection Include files.

//!@{

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>

#include "jcmacros.h"
#include "jcdebug.h"

#include "timecam-v.h"

//!@}

//!@subsection Macro-definitions, and constants.

//!@{

// now we define the list ITEM_LIST of possible items in
// the parameters file. note that they are functions of
// T(x). we will change T(x) to be the real item or the
// string of this item when needed

#define ITEM_LIST   /* LIST OF ITEMS IN THE PARAMETERS FILE */     \
T(input_file),      /* input file */                              \
T(output_file),     /* output file */                              \
T(data_file),       /* data file */                              \
T(diag_file),       /* diagnostic output file (ROOT format) */   \
T(root_file),       /* ROOT file */                              \
T(ct_file),         /* file with the characteristics of the CT */  \
T(ana_pixels),      /* size of the camera for parameters calculation */  \
T(nsb_on),          /* activates NSB simulation */ \
T(nsb_off),         /* de-activates NSB simulation */ \
T(nsb_mean),        /* mean value of NSB contribution per pixel */ \
T(threshold),       /* value of q0 for trigger */ \
T(tail_cut),        /* value of tail cut (t0) */ \
T(islands_on),      /* DO count islands */ \
T(islands_off),     /* do NOT count islands */ \
T(islands_cut),     /* value of islands cut (i0) */ \
T(seeds),           /* seeds for random number generation */ \
T(data_from_stdin), /* to read data from STDIN */ \
T(skip),            /* skip pathological showers */ \
T(read_phe_all),    /* id., but was processed with write_all_images */ \
T(read_phe),        /* read an already camera processed file */ \
T(write_all_images),/* write to file .phe ALL images (even w.o. trigger)*/ \
T(nowrite_McEvt),   /* nowrite to file .root McEvt infos */ \
T(write_McTrig),    /* write to file .root McTrig infos */ \
T(write_RawEvt),    /* write to file .root RawEvt infos */ \
T(write_all_data),  /* write to file .dat ALL image data */ \
T(select_energy),   /* energy range to read: only for .phe files */ \
T(trigger_radius),  /* trigger radius for the camera */ \
T(correction),      /* factor for correction in the pixel values */ \
T(end_file)         /* end of the parameters file */
  
#define T(x)  x             // define T() as the name as it is

enum ITEM_TYPE {
  ITEM_LIST
};

#undef T

#define T(x)  #x              // define T() as the string of x

const char *const ITEM_NAMES[] = {
  ITEM_LIST
};

#undef T

#define LINE_MAX_LENGTH  400
#define ITEM_MAX_LENGTH  40
#define PATH_MAX_LENGTH  120

// mean values of NSB contribution per pixel

static const float Mean_NSB_MAGIC = 5.0; //@< for MAGIC
static const float Mean_NSB_CT1 = 5.0;   //@< for CT1

//!@}

//!@subsection Prototypes of functions.

//!@{

//++
// prototypes
//--

void readparam(char * filename);
char *get_input_filename(void);
char *get_output_filename(void);
char *get_data_filename(void);
char *get_diag_filename(void);
char *get_root_filename(void);
char *get_ct_filename(void);
int get_nsb(float *n);
float get_threshold(void);
float get_tail_cut(void);
int get_islands_cut(int *n);
long int get_seeds(int n);
int get_ana_pixels(void);
void get_skip_showers( int *s ); 
int get_nskip_showers( void ); 
int get_data_from_stdin(void);
int get_read_phe(void);
int get_read_phe_all(void);
int get_write_all_images(void);
//
int get_write_McEvt(void);
int get_write_McTrig(void);
int get_write_RawEvt(void);
//
int get_write_all_data(void);
int get_select_energy(float *le, float *ue);
int get_trigger_radius(float *radius);
int get_correction(float *corr);
//!@}

//!@{

#endif // ! _creadparam_

//!@}

//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{

/*
 * $Log: not supported by cvs2svn $
 * Revision 1.1.1.1  1999/11/05 11:59:34  harald
 * This the starting point for CVS controlled further developments of the
 * camera program. The program was originally written by Jose Carlos. 
 * But here you can find a "rootified" version to the program. This means 
 * that there is no hbook stuff in it now. Also the output of the
 * program changed to the MagicRawDataFormat. 
 *
 * The "rootification" was done by Dirk Petry and Harald Kornmayer. 
 *
 * In the following you can see the README file of that version:
 *
 * ==================================================
 *
 * Fri Oct 22  1999   D.P.
 *
 * The MAGIC Monte Carlo System
 *
 * Camera Simulation Programme
 * ---------------------------
 *
 * 1) Description
 *
 * This version is the result of the fusion of H.K.'s
 * root_camera which is described below (section 2)
 * and another version by D.P. which had a few additional
 * useful features.
 *
 * The version compiles under Linux with ROOT 2.22 installed
 * (variable ROOTSYS has to be set).
 *
 * Compile as before simply using "make" in the root_camera
 * directory.
 *
 * All features of H.K.'s root_camera were retained.
 *
 * Additional features of this version are:
 *
 *   a) HBOOK is no longer used and all references are removed.
 *
 *   b) Instead of HBOOK, the user is given now the possibility of 
 *      having Diagnostic data in ROOT format as a complement
 *      to the ROOT Raw data.
 *
 *      This data is written to the file which is determined by
 *      the new input parameter "diag_file" in the camera parameter
 *      file.
 *
 *      All source code file belonging to this part have filenames
 *      starting with "MDiag".
 *
 *      The user can read the output file using the following commands
 *      in an interactive ROOT session:
 *
 *        	root [0] .L MDiag.so
 * 	root [1] new TFile("diag.root");
 * 	root [2] new TTreeViewer("T");
 * 	
 *      This brings up a viewer from which all variables of the
 *      TTree can be accessed and histogrammed. This example
 *      assumes that you have named the file "diag.root", that
 *      you are using ROOT version 2.22 or later and that you have
 *      the shared object library "MDiag.so" which is produced
 *      by the Makefile along with the executable "camera".
 *        
 *  !   The contents of the so-called diag file is not yet fixed.
 *  !   At the moment it is what J.C.G. used to put into the HBOOK
 *  !   ntuple. In future versions the moments calculation can be
 *  !   removed and the parameter list be modified correspondingly.
 *
 *   c) Now concatenated reflector files can be read. This is useful
 *      if you have run the reflector with different parameters but
 *      you want to continue the analysis with all reflector data
 *      going into ONE ROOT outputfile.
 *
 *      The previous camera version contained a bug which made reading 
 *      of two or more concatenated reflector files impossible.
 *
 *   d) The reflector output format was changed. It is now version
 *      0.4 .
 *      The change solely consists in a shortening of the flag
 *      definition in the file 
 *
 *            include-MC/MCCphoton.hxx  
 *
 *  !   IF YOU WANT TO READ REFLECTOR FORMAT 0.3, you can easily
 *  !   do so by recompiling camera with the previous version of
 *  !   include-MC/MCCphoton.hxx.
 *
 *      The change was necessary for saving space and better
 *      debugging. From now on, this format can be frozen.
 *
 *  !   For producing reflector output in the new format, you
 *  !   of course have to recompile your reflector with the
 *  !   new include-MC/MCCphoton.hxx .
 *
 *   e) A first version of the pixelization with the larger
 *      outer pixels is implemented. THIS IS NOT YET FULLY
 *      TESTED, but first rough tests show that it works
 *      at least to a good approximation.
 *
 *      The present version implements the camera outline
 *      with 18 "gap-pixels" and 595 pixels in total as
 *      shown in 
 *
 *         http://sarastro.ifae.es/internal/home/hardware/camera/numbering.ps
 *
 *      This change involved 
 *
 * 	(i) The file pixels.dat is no longer needed. Instead
 *           the coordinates are generated by the program itself
 *           (takes maybe 1 second). In the file 
 *
 * 		pixel-coords.txt
 *
 * 	  in the same directory as this README, you find a list
 *           of the coordinates generated by this new routine. It
 *           has the format
 *
 *               number   i   j   x  y  size-factor
 *
 *           where i and j are J.C.G.'s so called biaxis hexagonal
 *           coordinates (for internal use) and x and y are the
 *           coordinates of the pixel centers in the standard camera
 *           coordinate system in units of centimeters. The value
 *           of "size-factor" determines the linear size of the pixel
 *           relative to the central pixels. 
 *
 *         (ii) The magic.def file has two additional parameters
 *           which give the number of central pixels and the
 *           number of gap pixels
 *
 *         (iii) In camera.h and camera.cxx several changes were 
 *           necessary, among them the introduction of several
 *           new functions 
 *
 *      The newly suggested outline with asymmetric Winston cones
 *      will be implemented in a later version.
 *
 *   f) phe files can no longer be read since this contradicts
 *      our philosophy that the analysis should be done with other
 *      programs like e.g. EVITA and not with "camera" itself.
 *      This possibility was removed. 
 *
 *   g) ROOT is no longer invoked with an interactive interface.
 *      In this way, camera can better be run as a batch program and
 *      it uses less memory.
 *
 *   h) small changes concerning the variable "t_chan" were necessary in
 *      order to avoid segmentation faults: The variable is used as an
 *      index and it went sometimes outside the limits when camera
 *      was reading proton data. This is because the reflector files
 *      don't contain the photons in a chronological order and also
 *      the timespread can be considerably longer that the foreseen
 *      digitisation timespan. Please see the source code of camera.cxx
 *      round about line 1090.
 *
 *   j) several unused variables were removed, a few warning messages
 *      occur when you compile camera.cxx but these can be ignored at
 *      the moment.
 *
 * In general the program is of course not finished. It still needs
 * debugging, proper trigger simulation, simulation of the asymmetric
 * version of the outer pixels, proper NSB simulation, adaption of
 * the diag "ntuple" contents to our need and others small improvements.
 *
 * In the directory rfl-files there is now a file in reflector format 0.4
 * containing a single event produced by the starfiled adder. It has
 * a duration of 30 ns and represents the region around the Crab Nebula.
 * Using the enclosed input parameter file, camera should process this
 * file without problems.
 *
 * 2) The README for the previous version of root_camera
 *
 * README for a preliminary version of the 
 * root_camera program. 
 *
 * root_camera is based on the program "camera"of Jose Carlos
 * Gonzalez. It was changed in the way that only the pixelisation 
 * and the distibution of the phe to the FADCs works in a 
 * first version. 
 *
 * Using the #undef command most possibilities of the orignal 
 * program are switched of. 
 *
 * The new parts are signed by 
 *
 * - ROOT or __ROOT__ 
 *   nearly all  important codelines for ROOT output are enclosed 
 *   in structures like 
 *   #ifdef __ROOT__ 
 *   
 *     code 
 *
 *   #endif __ROOT__ 
 *
 *   In same case the new lines are signed by a comment with the word 
 *   ROOT in it. 
 *
 *   For timing of the pulse some variable names are changed. 
 *   (t0, t1, t  -->  t_ini, t_fin, t_1st, t_chan,...) 
 *   Look also for this changes. 
 *
 *   For the new root-file is also a change in readparm-files
 *
 *
 * - __DETAIL_TRIGGER__
 *
 *   This is for the implementation of the current work on trigger 
 *   studies. Because the class MTrigger is not well documented it 
 *   isnt a part of this tar file. Only a dummy File exists. 
 *
 *
 *
 * With all files in the archive, the root_camera program should run. 
 *
 * A reflector file is in the directory rfl-files
 *
 * ==================================================
 *
 * From now on, use CVS for development!!!!
 *
 *
 *
 * Revision 1.2  1999/10/22 15:01:29  petry
 * version sent to H.K. and N.M. on Fri Oct 22 1999
 *
 * Revision 1.1.1.1  1999/10/21 16:35:11  petry
 * first synthesised version
 *
 * Revision 1.7  1999/03/15  14:59:09  gonzalez
 * camera-1_1
 *
 * Revision 1.6  1999/03/02  09:56:13  gonzalez
 * *** empty log message ***
 *
 * Revision 1.5  1999/01/14  17:32:43  gonzalez
 * Added to camera the STDIN input option (data_from_input)
 *
 */

//!@}
//=EOF
