#ifndef __MFadc__
#define __MFadc__
//
//     class MFadc
//
//     implemented by Harald Kornmayer
//
//     This is a class to simulate the FADC. 
//     It assumes a special response of the PMT for one single Photo-electron. 
//     
//
//
#include <stream.h>
#include <math.h>

#include "TObject.h"
#include "TRandom.h"

#include "Mdefine.h" 

#include "MTriggerDefine.h"

class MMcEvt  ; 

//==========
//  MFadc 
//
//  The simulation of the Flash ADC system for the MAGIC teleskop is done with
//  this class. 
//  So all methods concerning the FADC System should be done inside this
//  class. 
//
//  The Idea is to (in)put the data of the photo electrons into the class and
//  generate the response (output) of the FADC to that input. Response means 
//  in this sense the ADC values of the different time slices for all pixels
//
//  The pixelisation is done by the camera program of Jose Carlos. 
//
//  This class is closly connected to the MTrigger classs. So some of the
//  values defined in MTriggerDefine.h are also used by this class. 
//
//  But a lot of other stuff is defined here. 
//
//  --> Frist of all the WIDTH of the time slice of one FADC slice 
//      this is 3.33333 nsec.
//
#define WIDTH_FADC_TIMESLICE   (50./15.)       //  this means 3.33 nsec 
//
//  --> Second the number of slices to fill in MFADC. This must by 
//      connected to the MTrigger class. The time of interest must be
//      equal in both classes. 
//
#define SLICES_MFADC           (TOTAL_TRIGGER_TIME / WIDTH_FADC_TIMESLICE)
//  --> The amount of ns before trigger that would be read from the ADC
//      in oder to show also the start of the pulse before the
//      the trigger time.
//
#define TIME_BEFORE_TRIGGER    10.
//  --> like the trigger the FADC value will also have a standard response
//      to one single Photo electron. This response is binned with smaller
//      bins. The WIDTH of that response function is done here. 
// 
#define SUBBINS     5.
#define WIDTH_RESPONSE_MFADC   (WIDTH_FADC_TIMESLICE /  SUBBINS )  
// 5 sub-bin in one FADC slice 
//
//  --> the number of Response slices
//
#define RESPONSE_SLICES_MFADC   45         
//
//
#define MFADC_RESPONSE_FWHM        5.0

//
//
#define MFADC_RESPONSE_AMPLITUDE   4.0
//
//
//
//
class MFadc {
 private:
  //
  //    then for all pixels the shape of all the analog signals 
  //
  Bool_t   used[CAMERA_PIXELS] ;  //  a boolean to indicated if the pixels is used in this event

  Float_t  sig[CAMERA_PIXELS][(Int_t) SLICES_MFADC] ; //  the analog signal for pixels
  UChar_t  output[CAMERA_PIXELS][FADC_SLICES];  //  the analog signal for pixels that is read after a trigger occurs.

  //
  //    first the data for the response function
  //
  Float_t fwhm_resp ;                      // fwhm of the phe_response function 
  Float_t ampl_resp ;                      // amplitude of the phe_response function (in mV)
  Float_t sing_resp[ RESPONSE_SLICES_MFADC ] ;   // the shape of the phe_response function 
  //
  //   RandomGenerator for the Electonic Noise
  //

  TRandom  *GenElec ; 


public:

  MFadc() ; 
  
  void Reset() ; 

  void Fill( Int_t, Float_t, Float_t  ) ;  

  void Baseline();

  void Offset( Float_t, Int_t );

  void ElecNoise() ; 

  void Scan() ; 

  void Scan(Float_t time) ;

  void GetResponse( Float_t *resp ) ; 

  void TriggeredFadc(Float_t time);

  void ShowSignal ( MMcEvt *McEvt , Float_t ) ; 

  Float_t GetFadcSignal(Int_t pixel, Int_t slice);

  Float_t GetAmplitude() {
    return ampl_resp ; 
  }

  Float_t GetFwhm() {
    return fwhm_resp ; 
  }

  
} ; 


#endif
