//=//////////////////////////////////////////////////////////////////////
//=
//= srreadparam            
//=
//= @file        srreadparam.cxx
//= @desc        Reading of parameters file
//= @author      O Blanch Bigas
//= @email       blanch@ifae.es
//= @date        Wed Feb 21 17:43:07 CET 2001
//=
//=----------------------------------------------------------------------
//=
//= Created: Wed Feb 21 17:43:07 CET 2001
//= Author:  Oscar Blanch Bigas
//= Purpose: Program for star response  simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: srreadparam.cxx,v $
//= $Revision: 1.1 $
//= $Author: magicsol $ 
//= $Date: 2001-02-23 10:13:44 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |srreadparam.cxx|.

/*!@"
  
  This section describes briefly the source code for the file
  |srreadparam.cxx|. This file is very closely related to the file
  |readparams.cxx| from the |reflector| program. Actually, this later
  file was the ancestror of the file you are looking at.

  All the defines it uses are located in the file |srreadparam.h|. In
  the first one we can see the definitions of the commands available
  for the parameters file. We describe these commands in a later
  section.
  
  @"*/

//!@subsection Includes and Global variables definition.

/*!@"
  
  All the defines are located in the file {\tt srreadparam.h}.

  @"*/
 
//!@{

#include "srreadparam.h"

//!@}

//!@subsection Definition of global variables.

/*!@"

  Here we define the global variables where the values from the
  parameters file are stored.
  
  @"*/

//!@{

static char Database_path[PATH_MAX_LENGTH];  //@< path to store the database
static float Simulated_Phe_l = 0.0;          //@< lower limit for phe loop
static float Simulated_Phe_u = 50.0;         //@< higher limit for phe loop
static float Simulated_Phe_p = 0.1;          //@< precision for phe loop
//!@}

//!@subsection The function |readparam()|.

//!-----------------------------------------------------------
// @name  srreadparam
//                                                
// @desc  read parameters from the stdin / parameters file
//
// @var   *filename  Name of the parameters file (NULL->STDIN)
//
// @date Wed Feb 21 17:43:07 CET 2001
//------------------------------------------------------------
// @function

//!@{ 
void 
readparam(char * filename)
{
  char line[LINE_MAX_LENGTH];    //@< line to get from the stdin
  char token[ITEM_MAX_LENGTH];   //@< a single token
  int i;                         //@< dummy counters
  ifstream ifile;

  ifile.open( filename );

  // loop till the "end" directive is reached
  int is_end = FALSE;
  while (! is_end) {          

    // get line from file or stdin
    if ( filename != NULL )
      ifile.getline(line, LINE_MAX_LENGTH);
    else
      cin.getline(line, LINE_MAX_LENGTH);

    // skip comments (start with '#')
    if (line[0] == '#')
      continue;

    // show user comments (start with '>')
    if (line[0] == '>') {
      cout << line << endl << flush;
      continue;
    }

    // look for each item at the beginning of the line
    for (i=0; i<=end_file; i++) 
      if (strstr(line, ITEM_NAMES[i]) == line)
        break;
        
    // if it is not a valid line, just ignore it
    if (i == end_file+1) {
      cerr << "Skipping unknown token in [" << line << "]\n";
      continue;
    }

    // case block for each directive
    switch ( i ) {

    case database_path:          //@< name of the output path
          
      // get the path for the outcoming database 
      sscanf(line, "%s %s", token, Database_path);

      break;

    case simulated_phe:       //@< limits for the phe loop
          
      // get energy range
      sscanf(line, "%s %f %f %f", token, &Simulated_Phe_l, &Simulated_Phe_u,
	     &Simulated_Phe_p);

      break;

    case end_file:            //@< end of the parameters file

      // show a short message
      is_end = TRUE;

      break;

    } // switch ( i ) 

  } // while (! is_end)

  // after the loop is finished, return to the main function
  return;
}
//!@}


//!-----------------------------------------------------------
// @name get_database_path
//                                                
// @desc get name of the ouput path
//
// @return   Name of the output path
//
// @date Wed Feb 21 17:57:19 CET 2001
//------------------------------------------------------------
// @function 

//!@{
char *
get_database_path(void)
{
  return (Database_path);
}
//!@}

//!-----------------------------------------------------------
// @name get_simulated_phe 
//
// @desc return limits for the phe loop
//
// @var *lphe  Lower limit in the phe loop
// @var *uphe  Higher limit in the phe loop
// @var *pphe  Precision in the phe loop
// @return  void
//
// @date Wed Feb 21 18:04:03 CET 2001
//------------------------------------------------------------
// @function 

//!@{
void
get_simulated_phe(float *lphe, float *uphe, float *pphe)
{
  *lphe =  Simulated_Phe_l;
  *uphe =  Simulated_Phe_u;
  *pphe =  Simulated_Phe_p;
}
//!@}


//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{
//
// $Log: not supported by cvs2svn $
//!@}

//=EOF
