#ifndef __MMcEvtBasic__
#define __MMcEvtBasic__

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

#include "MMcEvt.hxx"

//
// Version 1: 
// New container to keep the very basic informations on the
// original MC events produced by Corsika
//

class MMcEvtBasic : public MParContainer
{

private:
  MMcEvt::ParticleId_t fPartId;  // Type of particle
  Float_t              fEnergy;  // [GeV] Energy
  Float_t              fImpact;  // [cm] impact parameter

  // Telescope orientation (see TDAS 02-11 regarding the 
  // precise meaning of these angles):
  Float_t              fTelescopePhi;    // [rad]
  Float_t              fTelescopeTheta;  // [rad]

  
 public:
  MMcEvtBasic();
  
  MMcEvtBasic(MMcEvt::ParticleId_t, Float_t, Float_t, Float_t, Float_t);
  ~MMcEvtBasic(); 

  void Clear(Option_t *opt=NULL);

  void Fill(MMcEvt::ParticleId_t, Float_t, Float_t, Float_t, Float_t);

  void Print(Option_t *opt=NULL) const;

  MMcEvt::ParticleId_t GetPartId() const { return fPartId; }
  Float_t GetEnergy()  const { return fEnergy; }
  Float_t GetImpact()  const { return fImpact; }
  Float_t GetTelescopePhi() const { return fTelescopePhi; }
  Float_t GetTelescopeTheta() const { return fTelescopeTheta; }

  TString GetParticleName() const
  {
      switch (fPartId)
      {
      case MMcEvt::kGAMMA:    return "Gamma";
      case MMcEvt::kPOSITRON: return "Positron";
      case MMcEvt::kELECTRON: return "Electron";
      case MMcEvt::kANTIMUON: return "Anti-Muon";
      case MMcEvt::kMUON:     return "Muon";
      case MMcEvt::kPI0:      return "Pi-0";
      case MMcEvt::kNEUTRON:  return "Neutron";
      case MMcEvt::kPROTON:   return "Proton";
      case MMcEvt::kHELIUM:   return "Helium";
      case MMcEvt::kOXYGEN:   return "Oxygen";
      case MMcEvt::kIRON:     return "Iron";
      }

      return Form("Id:%d", fPartId);
  }

  TString GetParticleSymbol() const
  {
      switch (fPartId)
      {
      case MMcEvt::kGAMMA:    return "\\gamma";
      case MMcEvt::kPOSITRON: return "e^{+}";
      case MMcEvt::kELECTRON: return "e^{-}";
      case MMcEvt::kANTIMUON: return "\\mu^{+}";
      case MMcEvt::kMUON:     return "\\mu^{-}";
      case MMcEvt::kPI0:      return "\\pi^{0}";
      case MMcEvt::kNEUTRON:  return "n";
      case MMcEvt::kPROTON:   return "p";
      case MMcEvt::kHELIUM:   return "He";
      case MMcEvt::kOXYGEN:   return "O";
      case MMcEvt::kIRON:     return "Fe";
      }

      return Form("Id:%d", fPartId);
  }

  TString GetEnergyStr() const
  {
      if (fEnergy>1000)
          return Form("%.1fTeV", fEnergy/1000);

      if (fEnergy>10)
          return Form("%dGeV", (Int_t)(fEnergy+.5));

      if (fEnergy>1)
          return Form("%.1fGeV", fEnergy);

      return Form("%dMeV", (Int_t)(fEnergy*1000+.5));
  }


  void SetPartId(MMcEvt::ParticleId_t id)
    { fPartId = id; }

  void SetEnergy(Float_t Energy)
  { fEnergy=Energy; }              //Set Energy 
 
  void SetImpact(Float_t Impact) 
  { fImpact=Impact;}               //Set impact parameter

  void SetTelescopeTheta(Float_t Theta) { fTelescopeTheta=Theta; }

  void SetTelescopePhi  (Float_t Phi)   { fTelescopePhi=Phi; }

  ClassDef(MMcEvtBasic, 1) //Stores Basic Montecarlo Information of one event

};

#endif
