#ifndef MARS_MMcEvtBasic
#define MARS_MMcEvtBasic

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif


class MMcEvtBasic : public MParContainer
{
public:
    enum ParticleId_t
    {
        kUNDEFINED  = -1,
        kGAMMA      =  1,
        kPOSITRON   =  2,
        kELECTRON   =  3,
        kANTIMUON   =  5,
        kMUON       =  6,
        kPI0        =  7,
        kNEUTRON    = 13,
        kPROTON   =   14,
        kHELIUM   =  402,
        kOXYGEN   = 1608,
        kIRON     = 5626,
        kNightSky = 9999
    };

protected:
  ParticleId_t fPartId;  // Type of particle
  Float_t      fEnergy;  // [GeV] Energy
  Float_t      fImpact;  // [cm] impact parameter

  // Telescope orientation (see TDAS 02-11 regarding the 
  // precise meaning of these angles):
  Float_t      fTelescopePhi;    // [rad]
  Float_t      fTelescopeTheta;  // [rad]
  
public:
  MMcEvtBasic();
  MMcEvtBasic(ParticleId_t, Float_t, Float_t, Float_t, Float_t);

  // Getter
  ParticleId_t GetPartId() const { return fPartId; }

  Float_t GetEnergy()  const { return fEnergy; }
  Float_t GetImpact()  const { return fImpact; }

  Float_t GetTelescopePhi() const { return fTelescopePhi; }
  Float_t GetTelescopeTheta() const { return fTelescopeTheta; }

  static TString GetParticleName(Int_t id)
  {
      switch (id)
      {
      case kUNDEFINED:return "Undefined";
      case kGAMMA:    return "Gamma";
      case kPOSITRON: return "Positron";
      case kELECTRON: return "Electron";
      case kANTIMUON: return "Anti-Muon";
      case kMUON:     return "Muon";
      case kPI0:      return "Pi-0";
      case kNEUTRON:  return "Neutron";
      case kPROTON:   return "Proton";
      case kHELIUM:   return "Helium";
      case kOXYGEN:   return "Oxygen";
      case kIRON:     return "Iron";
      case kNightSky: return "NSB";
      }

      return Form("Id:%d", id);
  }

  static TString GetParticleSymbol(Int_t id)
  {
      switch (id)
      {
      case kUNDEFINED:return "N/A";
      case kGAMMA:    return "\\gamma";
      case kPOSITRON: return "e^{+}";
      case kELECTRON: return "e^{-}";
      case kANTIMUON: return "\\mu^{+}";
      case kMUON:     return "\\mu^{-}";
      case kPI0:      return "\\pi^{0}";
      case kNEUTRON:  return "n";
      case kPROTON:   return "p";
      case kHELIUM:   return "He";
      case kOXYGEN:   return "O";
      case kIRON:     return "Fe";
      case kNightSky: return "\\gamma_{NSB}";
      }

      return Form("Id:%d", id);
  }

  static TString GetEnergyStr(Float_t e)
  {
      if (e>=1000)
          return Form("%.1fTeV", e/1000);

      if (e>=10)
          return Form("%dGeV", (Int_t)(e+.5));

      if (e>=1)
          return Form("%.1fGeV", e);

      return Form("%dMeV", (Int_t)(e*1000+.5));
  }

  TString GetParticleSymbol() const
  {
      return GetParticleSymbol(fPartId);
  }

  TString GetParticleName() const
  {
      return GetParticleName(fPartId);
  }

  TString GetEnergyStr() const
  {
      return GetEnergyStr(fEnergy);
  }

  // Setter
  void SetPartId(ParticleId_t id) { fPartId = id; }
  void SetEnergy(Float_t Energy)  { fEnergy=Energy; }              //Set Energy
  void SetImpact(Float_t Impact)  { fImpact=Impact;}               //Set impact parameter

  void SetTelescopeTheta(Float_t Theta) { fTelescopeTheta=Theta; }
  void SetTelescopePhi  (Float_t Phi)   { fTelescopePhi=Phi; }

  void Fill(ParticleId_t, Float_t, Float_t, Float_t, Float_t);

  // TObject
  void Clear(Option_t *opt=NULL);
  void Print(Option_t *opt=NULL) const;

  ClassDef(MMcEvtBasic, 2) //Stores Basic Montecarlo Information of one event

};

#endif
