#include <iostream.h>

#include "TClonesArray.h"
#include "TString.h"
#include "TRandom.h"

#include "MRawPixel.h"

#include "Mdefine.h"

//==========
// MRawPixel 
//    
// One Event in the Camera of MAGIC consists of measurements of different
// Photomultiplier. The measurements are taken with FADCs. So the raw 
// informations of one pixel are the ADC values of the FADCs. MAGIC will
// measure the amplitude of one Photomulitplier in different branches 
// - one high and one low gain branch. The first is for small signals, the 
// other for big signals. With this method it is possible to increase the 
// dynamic range.  
//
// The Class MRawPixel is used to manage this raw information. 
//
// The data members are described in detail here: 
//
// ---------
//
// UShort_t     PixelId     ;   
//
// This is to identify the Pixel.
// The Camera consists of CAMERA_PIXELS Pixels. Each one has a specific Id.
// The center of the camera is zero and a spiral numbering is chosen.
//
// To reduce the amount of data, we store the information of the 
// low-gain-branch of the Fadcs only if there is a signal in. Then the 
// PixelId is:
//                   PixelId = 10000 + PixelId   . 
//
// For the high gain Fadc values the rule is:
//                   PixelId = PixelId . 
//
//
// ---------
//
// UChar_t      PixelStatus ;   
//
// The PixelStatus may contain information if the Pixel has triggered 
// this event. Also the kind of gain-branch (high or low) may be indicated 
// by one bit of the PixelStatus. etc
//
// ---------
//
// UChar_t      Fadc[ FADC_SLICES ]  ; 
//
// The values of FADC_SLICES fadc-slices. This is the information of the 
// measured ADC values for one branch (high gain or low gain). The typ of 
// branch is indicated in the usPixelNumber and in the PixelStatus.

ClassImp(MRawPixel) 



MRawPixel::MRawPixel() {
  //
  //  The default constructor sets all members to zero. 
  //
    
    PixelId     = 0 ; 
    PixelStatus = 0 ; 
    
    for (Int_t i = 0; i<FADC_SLICES; i++) 
      Fadc[i]   = 0 ; 
}


MRawPixel::MRawPixel(UShort_t PId) {
  //
  //  constructor II  overloaded
  //
  //  Parameter is the PixelId. All other data member are set to zero
  
  PixelId     = PId ; 
  PixelStatus = 0 ; 
  
  for (Int_t i = 0; i<FADC_SLICES; i++) 
    Fadc[i]   = 0 ; 
}


MRawPixel::MRawPixel(UShort_t usPId, UChar_t ucSt, UChar_t ucF[]) {
  //
  // constructor III overloaded 
  //
  // parameters are PixelId, Status and an array with Fadc numbers.
  
  PixelId     = usPId ; 
  PixelStatus = ucSt  ; 
  
  for (Int_t i = 0; i<FADC_SLICES; i++) 
    Fadc[i]   = ucF[i] ; 
}
MRawPixel::~MRawPixel(){
  //
  //  default destructor
  //
}

void MRawPixel::Clear() {
  //
  //  Sets the data member back to zero
  //
  PixelId     = 0 ; 
  PixelStatus = 0 ; 
  
  for (Int_t i = 0; i<FADC_SLICES; i++) 
    Fadc[i]   = 0 ; 
} 


void MRawPixel::Print() 
{
  //
  //  This member function prints all Data of one Pixel on screen. 
  //

  cout << endl << "PixId " << PixelId          ; 
  cout << " Stat " << (Int_t) PixelStatus << " --> "     ;

  for (Int_t i=0 ; i<FADC_SLICES ; i++ ) {
    cout<<" / " << (int) Fadc[i] ; 
  } 
}
  
UShort_t MRawPixel::GetPixelId() {
  //
  // returns back the PixelId of the Pixel
  //
  return PixelId;
}

UChar_t MRawPixel::GetFadcSlice( Int_t iSli ) {
  //
  // returns back the fadc content of the slice iSli 
  //
  return ( Fadc[iSli] ) ; 
}






