/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo, 02/2005 <mailto:moralejo@pd.infn.it>
!
!   Copyright: MAGIC Software Development, 2000-2005
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MMcEvtBasic
//
// This class contains the most basic MonteCarlo information
// with which an event has been generated
//
// Note: The azimuth fTelescopePhi angle in this and other MC classes 
// follow the convention in the Corsika program (see Corsika manual and
// TDAS 02-11). There, phi is the azimuth of the momentum vector of 
// particles, and is measured from the north direction, anticlockwise 
// (i.e, west is phi=90 degrees). When it refers to the telescope 
// orientation, it is the azimuth of a vector along the telescope axis, 
// going from the camera to the mirror. So, fTelescopeTheta=90, 
// fTelescopePhi = 0 means the telescope is pointing horizontally towards 
// South. 
//
//
// Version 1: 
//  New container to keep the very basic informations on the
//  original MC events produced by Corsika
//
// Version 2:
//  - added typedef for ParticleId_t from MMcEvt
//  - replaced MMcEvt::ParticleId_t by ParticleId_t
//
/////////////////////////////////////////////////////////////////////////////
#include "MMcEvtBasic.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MMcEvtBasic);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. Calls Clear()
//
MMcEvtBasic::MMcEvtBasic()
{
    fName  = "MMcEvtBasic";
    fTitle = "Basic event info from Monte Carlo";

    Clear();
}

// --------------------------------------------------------------------------
//
// Constructor. Use this to set all data members
//
// THIS FUNCTION IS FOR THE SIMULATION OLNY.
// DON'T USE THIS MEMBERFUNCTION IN THE ANALYSIS.
//
MMcEvtBasic::MMcEvtBasic(ParticleId_t usPId, Float_t fEner,
			 Float_t fImpa, Float_t fTPhii, Float_t fTThet)
{
    fName  = "MMcEvtBasic";
    fTitle = "Basic event info from Monte Carlo";

    Fill(usPId, fEner, fImpa, fTPhii, fTThet);
}

// --------------------------------------------------------------------------
//
//  Reset all values: Fill(kUNDEFINED, -1, -1, 0, 0)
//
void MMcEvtBasic::Clear(Option_t *opt)
{
    Fill(kUNDEFINED, -1, -1, 0, 0);
}

// --------------------------------------------------------------------------
//
// Fill all data members
//
void MMcEvtBasic::Fill(ParticleId_t usPId, Float_t fEner,
		       Float_t fImpa, Float_t fTPhii, Float_t fTThet)
{
    fPartId         = usPId;

    fEnergy         = fEner;
    fImpact         = fImpa;

    fTelescopePhi   = fTPhii;
    fTelescopeTheta = fTThet;
}

// --------------------------------------------------------------------------
//
// Print the contents of the container.
//
//  if you specify an option only the requested data members are printed:
//  allowed options are: id, energy, impact
//
void MMcEvtBasic::Print(Option_t *opt) const
{
    //
    //  print out the data member on screen
    //
    TString str(opt);
    if (str.IsNull())
    {
        *fLog << all << endl;
        *fLog << "Monte Carlo output:" << endl;
        *fLog << " Particle Id:    " << GetParticleName() << endl;
        *fLog << " Energy:         " << fEnergy << "GeV" << endl;
        *fLog << " Impactparam.:   " << fImpact/100 << "m" << endl;
        *fLog << endl;
        return;
    }
    if (str.Contains("id", TString::kIgnoreCase))
        *fLog << "Particle: " << GetParticleName() << endl;
    if (str.Contains("energy", TString::kIgnoreCase))
        *fLog << "Energy: " << fEnergy << "GeV" << endl;
    if (str.Contains("impact", TString::kIgnoreCase))
        *fLog << "Impact: " << fImpact << "cm" << endl;
}
