/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MMcPedestalCopy                                                        //
//                                                                         //
//  Input Containers:                                                      //
//   MMcFadcHeader                                                         //
//                                                                         //
//  Output Containers:                                                     //
//   MPedestalCam                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MMcPedestalCopy.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalCam.h"
#include "MRawRunHeader.h"
#include "MMcFadcHeader.hxx"

ClassImp(MMcPedestalCopy);

MMcPedestalCopy::MMcPedestalCopy(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalCopy";
    fTitle = title ? title : "Task to copy monte carlo pedestals into MPedestal Container";

    AddToBranchList("fPedestalMean");
    AddToBranchList("fElecNoise");
}

Bool_t MMcPedestalCopy::PreProcess(MParList *pList)
{
    MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (run)
    {
        if (run->GetRunType() != kRTMonteCarlo)
            return kTRUE;
    }

    const MMcFadcHeader *mcped = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");
    if (!mcped)
    {
        *fLog << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    MPedestalCam *pedcam = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!pedcam)
        return kFALSE;

    const int num = mcped->GetNumPixel();

    pedcam->InitSize(num);

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix = (*pedcam)[i];

        const Float_t pedest = mcped->GetPedestal(i);
        const Float_t pedrms = mcped->GetPedestalRms(i);

        const Float_t sigma  = pedest*sqrt(num);
        const Float_t sigrms = sigma/sqrt(num*2);

        pix.SetPedestal(pedest, sigma);
        pix.SetPedestalRms(pedrms, sigrms);
    }

    return kTRUE;
}

