/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MHillasCalc                                                            //
//                                                                         //
//  This is a task to calculate the Hillas parameters from each event      //
//                                                                         //
//  Input Containers:                                                      //
//   MCerPhotEvt, MGeomCam                                                 //
//                                                                         //
//  Output Containers:                                                     //
//   MHillas                                                               //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MHillasCalc.h"

#include "MParList.h"

#include "MHillas.h"
#include "MCerPhotEvt.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHillasCalc);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MHillasCalc::MHillasCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MHillasCalc";
    fTitle = title ? title : "Task to calculate Hillas parameters";
}

// --------------------------------------------------------------------------
//
// Check for a MCerPhotEvt object from which the Hillas are calculated.
// Try to find the Geometry conatiner. And try to find the output
// (Hillas) container or create one.
//
Bool_t MHillasCalc::PreProcess(MParList *pList)
{
    fCerPhotEvt = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fCerPhotEvt)
    {
        *fLog << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }

    fGeomCam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fGeomCam)
    {
        *fLog << dbginf << "MGeomCam (Camera Geometry) missing in Parameter List... aborting." << endl;
        return kFALSE;
    }

    fHillas = (MHillas*)pList->FindCreateObj("MHillas");
    if (!fHillas)
        return kFALSE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// If you want do complex descisions inside the calculations
// we must move the calculation code inside this function
//
// If the calculation wasn't sucessfull skip this event
//
Bool_t MHillasCalc::Process()
{
    return fHillas->Calc(*fGeomCam, *fCerPhotEvt) ? kTRUE : kCONTINUE;
}

