//
// This File contains the definition of the MGCoordinates-class
//
//   Author: Thomas Bretz
//   Version: V1.0 (1-8-2000)

#include "MGSkyPosition.h"

#include <iostream.h>  // cout

#include <TArc.h>
#include <TLine.h>
#include <TText.h>
#include <TList.h>
#include <TCanvas.h>

#include "coord.h"
#include "SlaPlanets.h"

void MGSkyPosition::InitPlanets()
{
    //
    // Magnitudes:
    // -----------
    //   Moon     -12.8
    //   Jupiter   -4.6
    //   Venus     -4.0
    //   Mars      -2.0
    //   Merkur     0.0
    //   Saturn     0.7
    //   Uranus     5.5
    //   Neptun     7.8
    //
    fSlalib = new SlaPlanets;

    //
    // Colors: black .. white:
    //
    //   1, 12, 13, 14, 15, 16, 17, 18, 19, 10
    //
    fMars = new TArc(0, 0, 1);
    fMars->SetFillStyle(1001);     // solid      (s. TAttFill)
    fMars->SetFillColor(13);       // light grey (s. TAttFill)
    fMars->SetLineColor(12);
    fMars->Draw();

    fVenus = new TArc(0, 0, 2);
    fVenus->SetFillStyle(1001);    // solid      (s. TAttFill)
    fVenus->SetFillColor(15);      // light grey (s. TAttFill)
    fVenus->SetLineColor(14);
    fVenus->Draw();

    fJupiter = new TArc(0, 0, 2);
    fJupiter->SetFillStyle(1001);  // solid      (s. TAttFill)
    fJupiter->SetFillColor(17);    // light grey (s. TAttFill)
    fJupiter->SetLineColor(16);
    fJupiter->Draw();

    fMoon = new TArc(0, 0, 3, 290, 70);
    fMoon->SetFillStyle(1001);     // solid      (s. TAttFill)
    fMoon->SetFillColor(19);       // light grey (s. TAttFill)
    fMoon->SetLineColor(15);
    fMoon->Draw();

    for (int i=0; i<4; i++)
    {
        fSunL[i] = new TLine(0,0,0,0);
        fSunL[i]->SetLineColor(10);    // white gray
        fSunL[i]->SetLineStyle(1);     // solid  (s. TAttLine)
        fSunL[i]->Draw();

        fList->Add(fSunL[i]);
    }

    fSun = new TArc(0, 0, 2);
    fSun->SetFillStyle(1001);     // solid      (s. TAttFill)
    fSun->SetFillColor(10);       // white      (s. TAttFill)
    fSun->SetLineColor(19);
    fSun->Draw();

    fList->Add(fMars);
    fList->Add(fVenus);
    fList->Add(fJupiter);
    fList->Add(fMoon);
    fList->Add(fSun);

    //
    // Reset 'static' positions of planets
    //
    for (int i=0; i<9; i++)
    {
        fX[i] = ~0;
        fY[i] = ~0;
    }
}

void MGSkyPosition::InitText()
{
    fText = new TText(105, 105, "");
    //fText->SetFillStyle(4000);  // transparent
    fText->SetTextAlign(33);  // right, top
    fText->SetTextColor(10);  // white
    fText->Draw();

    fList->Add(fText);
}

void MGSkyPosition::DrawCoordinateSystem()
{
    TArc arc;
    arc.SetFillStyle(4000);  // transparent
    arc.SetLineColor(13);    // dark gray
    arc.SetLineStyle(1);     // solid  (s. TAttLine)
    arc.DrawArc(0, 0, 90);
    arc.DrawArc(0, 0, 67.5);
    arc.DrawArc(0, 0, 45);
    arc.DrawArc(0, 0, 22.5);

    //
    // FIXME? Use TAxis?
    //
    TLine line;
    line.SetLineColor(1);  // black
    line.SetLineStyle(1);  // solid  (s. TAttLine)
    line.DrawLine(-95,   0, 95,  0);
    line.DrawLine(  0, -95,  0, 95);

    TText text;
    text.SetTextAlign(22);  // centered, centered (s.TAttText)
    text.SetTextColor(13);  // dark gray
    text.DrawText(0.71*22.5, 0.71*22.5, "22.5\xb0");
    text.DrawText(0.71*45.0, 0.71*45.0, "45\xb0");
    text.DrawText(0.71*67.5, 0.71*67.5, "67.5\xb0");
    text.DrawText(0.71*90.0, 0.71*90.0, "90\xb0");

    text.SetTextAlign(22);  // centered, centered (s.TAttText)
    text.SetTextColor(1);   // black

    text.DrawText(0,  102, "N (0\xb0)");
    text.DrawText(0, -102, "S (180\xb0)");

    text.SetTextAngle(270);
    text.DrawText( 102, 0, "E (90\xb0)");

    text.SetTextAngle(90);
    text.DrawText(-102, 0, "W (270\xb0)");
}

void MGSkyPosition::InitPosition()
{
    fLin1a = new TLine(0, 0, 0, 0);
    fLin1b = new TLine(0, 0, 0, 0);
    fLin2  = new TLine(0, 0, 0, 0);

    fLin1a->SetLineColor(5);  // yellow (s. TAttFill)
    fLin1b->SetLineColor(5);  // yellow
    fLin2 ->SetLineColor(10); // white
    fLin1a->SetLineStyle(1);  // solid  (s. TAttLine)
    fLin1b->SetLineStyle(2);  // dashed

    fLin1a->Draw();
    fLin1b->Draw();
    fLin2->Draw();

    fList->Add(fLin1a);
    fList->Add(fLin1b);
    fList->Add(fLin2);
}

MGSkyPosition::MGSkyPosition(const TGWindow* p, const UInt_t w)
    : MGEmbeddedCanvas("SkyPosition", p, w, 110)
{
    DrawCoordinateSystem();
    InitPlanets();
    InitPosition();
    InitText();
    InitCanvas();
}

MGSkyPosition::~MGSkyPosition()
{
    delete fSlalib;

    cout << "MGSkyPosition destroyed." << endl;
}

void MGSkyPosition::SetLin1(Float_t x, Float_t y)
{
    fLin1a->SetX2(x);
    fLin1a->SetY2(y);
    fLin1b->SetX2(-x);
    fLin1b->SetY2(-y);
}

void MGSkyPosition::SetLin2(Float_t x1, Float_t y1, Float_t x2, Float_t y2)
{
    fLin2->SetX1(x1);
    fLin2->SetX2(x2);
    fLin2->SetY1(y1);
    fLin2->SetY2(y2);
}

void MGSkyPosition::UpdatePosition(Float_t zd, Float_t az)
{
    static int X =~0;
    static int Y =~0;
    static int Rx=~0;
    static int Ry=~0;

    const float rad = D2PI*az/360.0;

    const float s = sin(rad);
    const float c = cos(rad);

    const float x = s*zd;
    const float y = c*zd;

    const int pixx = (int)(x/fPix);
    const int pixy = (int)(y/fPix);

    const int rx = (int)(s*fWidth/2.);
    const int ry = (int)(c*fWidth/2.);

    if (X==pixx && Y==pixy && Rx==rx && Ry==ry)
        return;

    X = pixx;
    Y = pixy;

    Rx = rx;
    Ry = ry;

    const float dx = s*4.;
    const float dy = c*4.;

    SetLin1(s*95., c*95.);
    SetLin2(x+dy, y-dx, x-dy, y+dx);

    SetModified();
}

void MGSkyPosition::UpdateText(Float_t x, Float_t y)
{
    static int X = ~0;
    static int Y = ~0;

    int xd = (int)floor(x);
    int yd = (int)floor(y);
    x *= 60.;
    y *= 60.;

    int fx = (int)floor(x*10.);
    int fy = (int)floor(y*10.);

    if (X==fx && Y==fy)
        return;

    X = fx;
    Y = fy;

    float xm = fmod(fabs(x), 60.);
    float ym = fmod(fabs(y), 60.);

    char txt[100];
    sprintf(txt, "Zd=%d\xb0%02.1f'\nAz=%d\xb0%02.1f'",
            xd, xm, yd, ym);

    fText->SetText(fText->GetX(), fText->GetY(), txt);

    SetModified();
}


void MGSkyPosition::Update(ZdAz &pos)
{
    //
    // calculate actual time for planet positions
    //
    fSlalib->SetMjd2Now();

    UpdatePlanet(kESun,     fSun);
    UpdatePlanet(kEMoon,    fMoon);
    UpdatePlanet(kEJupiter, fJupiter);
    UpdatePlanet(kEVenus,   fVenus);
    UpdatePlanet(kEMars,    fMars);

    UpdatePosition(pos.Zd(), pos.Az());
    UpdateText(pos.Zd(), pos.Az());

    UpdateCanvas();
}

void MGSkyPosition::UpdatePlanet(Int_t planet, TArc *arc)
{
    //
    // calculate actual position of planet
    //
    fSlalib->UpdatePlanetPos((ePlanets_t)planet);

    ZdAz zdaz = fSlalib->GetPlanetPos((ePlanets_t)planet);

    //
    // check whether we have to change the position where it is drawn
    //
    const float s = sin(zdaz.Az());
    const float c = cos(zdaz.Az());

    const float deg = 360.0/D2PI*zdaz.Zd();

    const float x = s*deg;
    const float y = c*deg;

    const int pixx = (int)(x/fPix);
    const int pixy = (int)(y/fPix);

    if (fX[planet]==pixx && fY[planet]==pixy)
        return;

    fX[planet] = pixx;
    fY[planet] = pixy;

    //
    // Set new position
    //
    arc->SetX1(x);
    arc->SetY1(y);

    SetModified();

    if (planet!=kESun)
        return;

    // cout << "Sun: x=" << x << " y=" << y;
    // cout << "   Zd=" << deg << " Az=" << 360./D2PI*zdaz.Az() << endl;

    fSunL[0]->SetX1(x-3.5); fSunL[0]->SetX2(x+3.5);
    fSunL[1]->SetX1(x-2.5); fSunL[1]->SetX2(x+2.5);
    fSunL[2]->SetX1(x-2.5); fSunL[2]->SetX2(x+2.5);
    fSunL[3]->SetX1(x);     fSunL[3]->SetX2(x);

    fSunL[0]->SetY1(y);     fSunL[0]->SetY2(y);
    fSunL[1]->SetY1(y-2.5); fSunL[1]->SetY2(y+2.5);
    fSunL[2]->SetY1(y+2.5); fSunL[2]->SetY2(y-2.5);
    fSunL[3]->SetY1(y-3.5); fSunL[3]->SetY2(y+3.5);
}
