///////////////////////////////////////////////////////////////////////
//
// MHHillas
//
// This class contains histograms for every Hillas parameter
//
///////////////////////////////////////////////////////////////////////

#include "MHHillas.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MHillas.h"

ClassImp(MHHillas);

// --------------------------------------------------------------------------
//
// Setup four histograms for Alpha, Width, Length and Dist
//
MHHillas::MHHillas (const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillas" ;
    fTitle = title ? title : "Container for Hillas histograms" ;

    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fAlpha  = new TH1F("\\alpha [\\circ]", "Alpha of Hillas",   90, 0,  90);
    fWidth  = new TH1F("Width [mm]",  "Width of Hillas",  100, 0, 300);
    fLength = new TH1F("Length [mm]", "Length of Hillas", 100, 0, 300);
    fDist   = new TH1F("Dist [mm]",   "Dist of Hillas",   100, 0, 300);

    fAlpha->SetDirectory(NULL);
    fLength->SetDirectory(NULL);
    fDist->SetDirectory(NULL);
    fWidth->SetDirectory(NULL);

    fAlpha->GetXaxis()->SetTitle("\\alpha [\\circ]");
    fLength->GetXaxis()->SetTitle("Length [mm]");
    fDist->GetXaxis()->SetTitle("Dist [mm]");
    fWidth->GetXaxis()->SetTitle("Width [mm]");

    fAlpha->GetYaxis()->SetTitle("Counts");
    fLength->GetYaxis()->SetTitle("Counts");
    fDist->GetYaxis()->SetTitle("Counts");
    fWidth->GetYaxis()->SetTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHHillas::~MHHillas()
{
    delete fAlpha;
    delete fWidth;
    delete fLength;
    delete fDist;
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
void MHHillas::Fill(const MParContainer *par)
{
    const MHillas &h = *(MHillas*)par;

    fAlpha ->Fill(fabs(h.GetAlpha()));
    fWidth ->Fill(h.GetWidth());
    fLength->Fill(h.GetLength());
    fDist  ->Fill(h.GetDist());
}

// --------------------------------------------------------------------------
//
// Draw clones of all four histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHHillas::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("Hillas", "Histograms of Hillas Parameters");
    c->Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    c->cd(1);
    fAlpha->DrawCopy();

    c->cd(2);
    fLength->DrawCopy();

    c->cd(3);
    fDist->DrawCopy();

    c->cd(4);
    fWidth->DrawCopy();

    c->Modified();
    c->Update();

    return c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the four histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillas::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas("Hillas", "Histograms of Hillas Parameters");

    gPad->Divide(2,2);

    gPad->cd(1);
    fAlpha->Draw();

    gPad->cd(2);
    fLength->Draw();

    gPad->cd(3);
    fDist->Draw();

    gPad->cd(4);
    fWidth->Draw();

    gPad->Modified();
    gPad->Update();
}
