/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  01/2002 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//   MFAlpha                                                               //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MFAlpha.h"

#include <math.h>

#include "MParList.h"
#include "MLog.h"
#include "MLogManip.h"

#include "MHillasSrc.h"

ClassImp(MFAlpha);


// --------------------------------------------------------------------------
//
MFAlpha::MFAlpha(const char *cname, const char type, const Float_t val,
                 const char *name, const char *title) : fHillas(NULL)
{
    fContName = cname;
    Init(type, val, name, title);
}

// --------------------------------------------------------------------------
//
MFAlpha::MFAlpha(const MHillasSrc *hillas, const char type, const Float_t val,
                 const char *name, const char *title) : fHillas(hillas)
{
    Init(type, val, name, title);
}

// --------------------------------------------------------------------------
//
void MFAlpha::Init(const char type, const Int_t val,
                   const char *name, const char *title)
{
    fName  = name  ? name  : "MFAlpha";
    fTitle = title ? title : "Filter using the alpha angle";

    fFilterType = (type=='<' ? kELowerThan : kEGreaterThan);

    if (type!='<' && type!='>')
        *fLog << warn << dbginf << "Warning: Neither '<' nor '>' specified... using '>'." << endl;

    fValue = val;

    AddToBranchList(Form("%s.fAlpha", (const char*)fContName));
}

// --------------------------------------------------------------------------
//
Bool_t MFAlpha::IsExpressionTrue() const
{
    return fResult;
}

// --------------------------------------------------------------------------
//
Bool_t MFAlpha::PreProcess(MParList *pList)
{
    if (fHillas)
        return kTRUE;

    fHillas = (MHillasSrc*)pList->FindObject(fContName);
    if (fHillas)
        return kTRUE;

    *fLog << err << dbginf << fContName << " [MHillas] not found... aborting." << endl;
    return kFALSE;
}

// --------------------------------------------------------------------------
//
Bool_t MFAlpha::Process()
{
    const Float_t alpha = fabs(fHillas->GetAlpha());

    switch (fFilterType)
    {
    case kELowerThan:
        fResult = (alpha < fValue);
        break;
    case kEGreaterThan:
        fResult = (alpha > fValue);
        break;
    }

    return kTRUE;
}

