/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MHillasCalc                                                            //
//                                                                         //
//  This is a task to calculate the Hillas parameters from each event      //
//                                                                         //
//  Input Containers:                                                      //
//   MCerPhotEvt, MGeomCam                                                 //
//                                                                         //
//  Output Containers:                                                     //
//   MHillas                                                               //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MHillasCalc.h"

#include "MParList.h"

#include "MHillas.h"
#include "MCerPhotEvt.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHillasCalc);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MHillasCalc::MHillasCalc(const char *hil, const char *name, const char *title)
{
    fName  = name  ? name  : "MHillasCalc";
    fTitle = title ? title : "Task to calculate Hillas parameters";

    fHilName = hil;
}

// --------------------------------------------------------------------------
//
// Check for a MCerPhotEvt object from which the Hillas are calculated.
// Try to find the Geometry conatiner. And try to find the output
// (Hillas) container or create one.
//
Bool_t MHillasCalc::PreProcess(MParList *pList)
{
    fCerPhotEvt = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fCerPhotEvt)
    {
        *fLog << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }

    fGeomCam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fGeomCam)
    {
        *fLog << dbginf << "MGeomCam (Camera Geometry) missing in Parameter List... aborting." << endl;
        return kFALSE;
    }

    fHillas = (MHillas*)pList->FindCreateObj(fHilName);
    if (!fHillas)
        return kFALSE;

    memset(fErrors, 0, sizeof(fErrors));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// If you want do complex descisions inside the calculations
// we must move the calculation code inside this function
//
// If the calculation wasn't sucessfull skip this event
//
Bool_t MHillasCalc::Process()
{
    const Int_t rc = fHillas->Calc(*fGeomCam, *fCerPhotEvt);
    if (rc<0 || rc>4)
    {
        *fLog << err << dbginf << "MHillas::Calc returned unknown error code!" << endl;
        return kFALSE;
    }
    fErrors[rc]++;

    return rc==0 ? kTRUE : kCONTINUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the hillas calculation. The percentage
//  is calculated with respect to the number of executions of this task.
//
Bool_t MHillasCalc::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fErrors[1] << " (" << setw(3) << (int)(fErrors[1]*100/GetNumExecutions()) << "%) Evts skipped due to: Event has less than 3 pixels" << endl;
    *fLog << " " << setw(7) << fErrors[2] << " (" << setw(3) << (int)(fErrors[2]*100/GetNumExecutions()) << "%) Evts skipped due to: Calculated Size == 0" << endl;
    *fLog << " " << setw(7) << fErrors[3] << " (" << setw(3) << (int)(fErrors[3]*100/GetNumExecutions()) << "%) Evts skipped due to: Number of used pixels < 3" << endl;
    *fLog << " " << setw(7) << fErrors[4] << " (" << setw(3) << (int)(fErrors[4]*100/GetNumExecutions()) << "%) Evts skipped due to: CorrXY==0" << endl;
    *fLog << " " << fErrors[0] << " (" << (int)(fErrors[0]*100/GetNumExecutions()) << "%) Evts survived Hillas calculation!" << endl;
    *fLog << endl;

    return kTRUE;
}
