/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHTimeDiffTheta                                                         //
//                                                                          //
//  calculates the 2D-histogram   time-difference vs. Theta                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHTimeDiffTheta.h"

#include <TCanvas.h>

#include "MTime.h"
#include "MMcEvt.hxx"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHTimeDiffTheta);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram.
//
MHTimeDiffTheta::MHTimeDiffTheta(const char *name, const char *title)
    : fLastTime(0), fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHTimeDiffTheta";
    fTitle = title ? title : "2-D histogram in Theta and time difference";

    fHist.SetDirectory(NULL);

    fHist.SetXTitle("\\Delta t [s]");
    fHist.SetYTitle("\\Theta [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histogram
//
Bool_t MHTimeDiffTheta::SetupFill(const MParList *plist)
{
   fTime = (MTime*)plist->FindObject("MTime");
   if (!fTime)
   {
       *fLog << err << dbginf << "MTime not found... aborting." << endl;
       return kFALSE;
   }

   fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
   if (!fMcEvt)
   {
       *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
       return kFALSE;
   }

   const MBinning* binsdtime = (MBinning*)plist->FindObject("BinningTimeDiff");
   const MBinning* binstheta = (MBinning*)plist->FindObject("BinningTheta");
   if (!binstheta || !binsdtime)
   {
       *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsdtime, binstheta);

   return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw a copy of the histogram
//
TObject *MHTimeDiffTheta::DrawClone(Option_t *opt) const
{
    TCanvas &c = *MakeDefCanvas("DiffTimeTheta", "Distrib of \\Delta t, Theta");

    c.Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // FIXME: ProjectionX,Y is not const within root
    //

    TH1D *h;

    c.cd(1);
    h = ((TH2*)&fHist)->ProjectionX("ProjX-Theta", -1, 9999, "E");

    h->SetTitle("Distribution of \\Delta t [s]");
    h->SetXTitle("\\Delta t [s]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);;
    gPad->SetLogy();

    c.cd(2);
    h = ((TH2*)&fHist)->ProjectionY("ProjY-timediff", -1, 9999, "E");

    h->SetTitle("Distribution of \\Theta [\\circ]");
    h->SetXTitle("\\Theta [\\circ]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);;

    c.cd(3);
    ((TH2*)&fHist)->DrawCopy(opt);

    c.Modified();
    c.Update();

    return &c;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHTimeDiffTheta::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("DiffTimeTheta", "Distrib of Delta t, Theta");

    TH1D *h;

    gPad->Divide(2,2);

    gPad->cd(1);
    h = fHist.ProjectionX("ProjX_Theta", -1, 9999, "E");

    h->SetTitle("Distribution of \\Delta t [s]");
    h->SetXTitle("\\Delta t [s]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);;
    gPad->SetLogy();

    gPad->cd(2);
    h = fHist.ProjectionY("ProjY_timediff", -1, 9999, "E");

    h->SetTitle("Distribution of \\Theta [\\circ]");
    h->SetXTitle("\\Theta [\\circ]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);;

    gPad->cd(3);
    fHist.DrawCopy(opt);

    gPad->Modified();
    gPad->Update();

}

// --------------------------------------------------------------------------
//
//  Fill the histogram
//
Bool_t MHTimeDiffTheta::Fill(const MParContainer *par)
{
    const Int_t time = fTime->GetTimeLo();

    fHist.Fill(0.0001*(time-fLastTime), fMcEvt->GetTheta()*kRad2Deg);
    fLastTime = time;

    return kTRUE;
}

