/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz <mailto:tbretz@astro.uni-wuerzburg.de>, 10/2002
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MContinue
//
// Does nothing than return kCONTINUE in the Process-function
// (use with filters). For more details see the description of the
// constructors.
//
/////////////////////////////////////////////////////////////////////////////
#include "MContinue.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MF.h"
#include "MParList.h"
#include "MTaskList.h"

ClassImp(MContinue);

// --------------------------------------------------------------------------
//
// Use this constructor if a rule (see MF for more details) shell be used.
// MContinue will create a MF object and use it as a filter for the
// instance. The MF-Task is added to the tasklist in front of the MContinue
// instance and also automatically deleted, eg.
//   MContinue cont("MHillas.fSize<20");
//   tasklist.AddToList(&cont);
// would skip all events which fullfill "MHillas.fSize<20" from this point
// in the tasklist.
//
MContinue::MContinue(const TString rule, const char *name, const char *title)
{
    fName  = name  ? name  : "MContinue";
    fTitle = title ? title : "Task returning kCONTINUE";

    if (rule.IsNull())
        return;

    SetBit(kIsOwner);

    MTask::SetFilter(new MF(rule, TString("MF(")+fName+")"));
}

// --------------------------------------------------------------------------
//
// Use this if you have a filter. Would be the same as if you would call:
//   MContinue cont;
//   cont.SetFilter(f);
//
MContinue::MContinue(MFilter *f, const char *name, const char *title)
{
    fName  = name  ? name  : "MContinue";
    fTitle = title ? title : "Task returning kCONTINUE";

    SetFilter(f);
}

// --------------------------------------------------------------------------
//
//  Delete the filter if it was created automatically
//
MContinue::~MContinue()
{
    if (TestBit(kIsOwner))
        delete GetFilter();
}

// --------------------------------------------------------------------------
//
//  In case the filter was created automatically, PreProcess tries to find
//  the tasklist MTaskList, adds the filter before this instance to the
//  tasklist and preprocesses the filter.
//
Bool_t MContinue::PreProcess(MParList *list)
{
    if (!TestBit(kIsOwner))
        return kTRUE;

    MTaskList *tlist = (MTaskList*)list->FindObject("MTaskList");
    if (!tlist)
    {
        *fLog << err << dbginf << "ERROR - Tasklist 'MTaskList' not found... abort." << endl;
        return kFALSE;
    }

    if (!GetFilter())
    {
        *fLog << err << dbginf << "Unknown fatal Error! (fFilter=NULL?!?)" << endl;
        return kFALSE;
    }

    if (!tlist->AddToListBefore(GetFilter(), this))
    {
        *fLog << err << dbginf << "ERROR - Adding filter before MContinue... abort." << endl;
        return kFALSE;
    }

    return GetFilter()->CallPreProcess(list);
}

