/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHTimeDiffTime                                                          //
//                                                                          //
//  calculates the 2D-histogram   time-difference vs. time                  //
//                                                                          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHTimeDiffTime.h"

#include <TCanvas.h>

#include "MTime.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHTimeDiffTime);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram
//
MHTimeDiffTime::MHTimeDiffTime(const char *name, const char *title)
    : fLastTime(0), fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHTimeDiffTime";
    fTitle = title ? title : "2-D histogram in time and time difference";

    fHist.SetDirectory(NULL);

    fHist.SetXTitle("\\Delta t [s]");
    fHist.SetYTitle("t [s]");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histogram
//
Bool_t MHTimeDiffTime::SetupFill(const MParList *plist)
{
    fTime = (MTime*)plist->FindObject("MTime");
    if (!fTime)
    {
        *fLog << err << dbginf << "MTime not found... aborting." << endl;
        return kFALSE;
    }

    const MBinning* binsdtime = (MBinning*)plist->FindObject("BinningTimeDiff");
    const MBinning* binstime  = (MBinning*)plist->FindObject("BinningTime");
    if (!binstime || !binsdtime)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHist, binsdtime, binstime);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw a copy of the histogram
//
TObject *MHTimeDiffTime::DrawClone(Option_t *opt) const
{

    TCanvas &c = *MakeDefCanvas("DiffTimeTime", "Distrib of dt and t");

    c.Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    TH1D *h;

    c.cd(1);
    h = ((TH2*)&fHist)->ProjectionX("ProjX-sumtime", -1, 9999, "E");

    h->SetTitle("Distribution of \\Delta t [s]");
    h->SetXTitle("\\Delta t [s]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogy();

    c.cd(2);
    h = ((TH2*)&fHist)->ProjectionY("ProjY-sumtimediff", -1, 9999, "E");

    h->SetTitle("Distribution of time [s]");
    h->SetXTitle("time [s]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);

    c.cd(3);
    ((TH2*)&fHist)->DrawCopy(opt);

    c.Modified();
    c.Update();

    return &c;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHTimeDiffTime::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("DiffTimeTime", "Distrib of \\Delta t, time");

    gPad->Divide(2,2);

    TH1D *h;

    gPad->cd(1);
    h = fHist.ProjectionX("ProjX_sumtime", -1, 9999, "E");

    h->SetTitle("Distribution of \\Delta t [s]");
    h->SetXTitle("\\Delta t [s]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogy();

    gPad->cd(2);
    h = fHist.ProjectionY("ProjY_sumtimediff", -1, 9999, "E");

    h->SetTitle("Distribution of time [s]");
    h->SetXTitle("time [s]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(3);
    fHist.DrawCopy(opt);

    gPad->Modified();
    gPad->Update();

}

// --------------------------------------------------------------------------
//
//  Fill the histogram
//
Bool_t MHTimeDiffTime::Fill(const MParContainer *par)
{
    const Double_t time = 200e-9*fTime->GetTimeLo() + fTime->GetTimeHi();

    fHist.Fill(time-fLastTime, time);
    fLastTime = time;

    return kTRUE;
}




