/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Robert Wagner 10/2002 <mailto:magicsoft@rwagner.de>
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHSigmaPixel                                                            //
//                                                                          //
//  2D-Histogram pedestal sigma vs pixel number                             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHSigmaPixel.h"

#include <TCanvas.h>

#include <math.h>

#include "MPedestalCam.h"
#include "MPedestalPix.h"
#include "MSigmabar.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHSigmaPixel);

// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram. 
//
MHSigmaPixel::MHSigmaPixel(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHSigmaPixel";
    fTitle = title ? title : "2-D histogram in sigma and pixel";

    fHist.SetDirectory(NULL);

    fHist.SetTitle("Sigma vs pixel #");
    fHist.SetXTitle("pixel #");
    fHist.SetYTitle("\\sigma");
    fHist.SetZTitle("N");
}

// --------------------------------------------------------------------------
//
// Set binnings and prepare filling of the histogram. The binning for the
// pixel axis is derived automagically from the MPedestalCam container
// expected to be found in the MParList *plist.
// 
Bool_t MHSigmaPixel::SetupFill(const MParList *plist)
{
  fPedestalCam = (MPedestalCam*)plist->FindObject("MPedestalCam");
  if (!fPedestalCam)
    {
      *fLog << err << dbginf << "MHSigmaPixel: MPedestalCam not found... aborting." << endl;
      return kFALSE;
    }
  
  MBinning* binssigma = (MBinning*)plist->FindObject("BinningSigma");
  MBinning* binspixel = new MBinning();
  binspixel->SetEdges(fPedestalCam->GetSize(), -0.5, -0.5+fPedestalCam->GetSize());
  
  if (!binssigma)
    {
      *fLog << err << dbginf << "MHSigmaPixel: BinningSigma not found... aborting." << endl;
      return kFALSE;      
   }

   SetBinning(&fHist, binspixel, binssigma);

   fHist.Sumw2(); 

   return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histogram
// 
Bool_t MHSigmaPixel::Fill(const MParContainer *par)
{
  MPedestalCam &ped = *(MPedestalCam*)par;
    for (Int_t i=0;i<(ped.GetSize());i++)
    {
      const MPedestalPix pix = ped[i];       
      fHist.Fill(i, pix.GetSigma());
    }
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
// 
void MHSigmaPixel::Draw(Option_t *opt)
{
  //gStyle->SetOptStat(1000);
  if (!gPad)
    MakeDefCanvas("SigmaPixel", fTitle);
  
  fHist.Draw(opt);
  
  gPad->Modified();
  gPad->Update();
}

// --------------------------------------------------------------------------
//
// Draw copies of the histogram
// 
TObject *MHSigmaPixel::DrawClone(Option_t *opt) const
{
  //gStyle->SetOptStat(1000);
  TCanvas &c = *MakeDefCanvas("SigmaPixel", fTitle);
  
  ((TH2&)fHist).DrawCopy(opt);
  
  c.Modified();
  c.Update();
  
  return &c;
}

