/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Hengstebeck 3/2003 <mailto:hengsteb@alwa02.physik.uni-siegen.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHRanForest
//
// This histogram shows the evolution of the standard deviation 
// of est. hadronness as the number of trees increases. 
// It helps you to find out how many trees are really needed in g/h-sep.
//
////////////////////////////////////////////////////////////////////////////
#include "MHRanForest.h"

#include <TPad.h>
#include <TGraph.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TMarker.h>

#include "MParList.h"
#include "MBinning.h"
#include "MRanForest.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcEvt.hxx"

ClassImp(MHRanForest);

// --------------------------------------------------------------------------
//
// Setup histograms, nbins is the number of bins used for the evaluation.
// The default is 100 bins.
//
MHRanForest::MHRanForest(Int_t nbins, const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHRanForest";
    fTitle = title ? title : "Histogram showing Standard deviation of estimated hadronness";

    fGraphSigma = new TGraph;
    fGraphSigma->SetTitle("Evolution of Standard deviation of estimated hadronness in tree combination");
    fGraphSigma->SetMaximum(1);
}

// --------------------------------------------------------------------------
//
// Delete the histograms.
//
MHRanForest::~MHRanForest()
{
    delete fGraphSigma;
}

// --------------------------------------------------------------------------
//
// Setup Filling of the histograms. It needs:
//  MMcEvt and MRanForest
//
Bool_t MHRanForest::SetupFill(const MParList *plist)
{
    fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    fRanForest = (MRanForest*)plist->FindObject("MRanForest");
    if (!fRanForest)
    {
        *fLog << err << dbginf << "MRanForest not found... aborting." << endl;
        return kFALSE;
    }

    fSigma.Set(fRanForest->GetNumTrees());
    fNumEvent=0;
    return kTRUE;
}

// --------------------------------------------------------------------------
//
//
Bool_t MHRanForest::Fill(const MParContainer *par)
{
    fNumEvent++;
    Double_t hest=0;
    Double_t htrue=fMcEvt->GetPartId()==kGAMMA ? 0. : 1.;

    Int_t ntrees=fRanForest->GetNumTrees();

    for (Int_t i=0;i<ntrees;i++)
    {
        for(Int_t j=0;j<=i;j++)
            hest+=fRanForest->GetTreeHad(j);

        hest/=i+1;
        fSigma[i]+=(htrue-hest)*(htrue-hest);
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Finalize the histogram:
// calculate standard deviation and set histogram max and min
//
Bool_t MHRanForest::Finalize()
{
    Int_t n = fSigma.GetSize();

    fGraphSigma->Set(n);

    Stat_t max=0.;
    Stat_t min=0.;
    for (Int_t i=0; i<n; i++)
    {
        Stat_t ip = i+1.;
	fSigma[i] = TMath::Sqrt(fSigma[i]/Stat_t(fNumEvent));
        Stat_t ig = fSigma[i];
        max=TMath::Max(max,ig);
        min=TMath::Min(min,ig);
        fGraphSigma->SetPoint(i,ip,ig);
    }
    fGraphSigma->SetMaximum(1.05*max);
    fGraphSigma->SetMinimum(0.95*min);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw clone of histogram
//
TObject *MHRanForest::DrawClone(Option_t *opt) const
{
    if (fGraphSigma->GetN()==0)
        return NULL;

    TCanvas &c = *MakeDefCanvas("RanForest", fTitle);
    gROOT->SetSelectedPad(NULL);

    //gStyle->SetOptStat(10);
    TGraph &g = (TGraph&)*fGraphSigma->DrawClone("AL");
    g.SetBit(kCanDelete);
    gPad->Modified();
    gPad->Update();
    if (g.GetHistogram())
    {
        g.GetXaxis()->SetRangeUser(0, fNumEvent);
        g.GetXaxis()->SetTitle("Number of Trees");
        g.GetYaxis()->SetTitle("Standard deviation of estimated hadronness");
        g.SetMarkerStyle(kFullDotMedium);
        gPad->Modified();
        gPad->Update();
        //g.Draw("P");
    }
    gPad->SetGrid();

    return &c;
}

// --------------------------------------------------------------------------
//
// Draw histogram. (For the Meaning see class description)
//
void MHRanForest::Draw(Option_t *)
{
   if (fGraphSigma->GetN()==0)
        return;

   if (!gPad)
        MakeDefCanvas("RanForest", fTitle);

    //gStyle->SetOptStat(10);
    fGraphSigma->Draw("ALP");
    gPad->Modified();
    gPad->Update();
    if (fGraphSigma->GetHistogram())
    {
        fGraphSigma->GetXaxis()->SetRangeUser(0, 1);
        fGraphSigma->GetXaxis()->SetTitle("Number of Trees");
        fGraphSigma->GetYaxis()->SetTitle("Standard deviation of estimated hadronness");

        fGraphSigma->SetMarkerStyle(kFullDotSmall);
        //fGraphSigma->Draw("P");
        gPad->Modified();
        gPad->Update();
    }    
}
