/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  04/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MFCT1SelStandard                                                       //
//                                                                         //
//  This is a class to evaluate the Standard Cuts                          //
//                                                                         //
//  to be called after the calculation of the image parameters             //
//               before the g/h separation                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MFCT1SelStandard.h"

#include "MParList.h"

#include "MHillas.h"
#include "MHillasExt.h"
#include "MHillasSrc.h"
#include "MCerPhotEvt.h"
#include "MMcEvt.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MFCT1SelStandard);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MFCT1SelStandard::MFCT1SelStandard(const char *HilName, const char *HilSrcName,
                                   const char *name, const char *title)
{
    fName  = name  ? name  : "MFCT1SelStandard";
    fTitle = title ? title : "Class to evaluate the Standard Cuts";

    fHilName    = HilName;
    fHilSrcName = HilSrcName;

    // default values of cuts
    SetCuts(92, 4, 60, 0.4, 1.05, 0.0, 0.0);
}

// --------------------------------------------------------------------------
//
// Set the values for the cuts 
// 
//
void MFCT1SelStandard::SetCuts(Float_t usedpixelsmax, Float_t corepixelsmin,
                          Float_t sizemin, Float_t distmin, Float_t distmax,
                          Float_t lengthmin, Float_t widthmin)
{ 
  fUsedPixelsMax = usedpixelsmax; 
  fCorePixelsMin = corepixelsmin;
  fSizeMin       = sizemin; 
  fDistMin       = distmin; 
  fDistMax       = distmax;
  fLengthMin     = lengthmin; 
  fWidthMin      = widthmin;

  *fLog << inf << "MFCT1SelStandard cut values : fUsedPixelsMax, fCorePixelsMin = "
        << fUsedPixelsMax << ",  " << fCorePixelsMin << endl;
  *fLog << inf << "     fSizeMin, fDistMin, fDistMax = " << fSizeMin 
        << ",  " << fDistMin << ",  " << fDistMax << endl; 
  *fLog << inf << "     fLengthMin, fWidthMin = " << fLengthMin 
        << ",  " << fWidthMin << endl; 
}

// --------------------------------------------------------------------------
//
// 
// 
// 
//
Bool_t MFCT1SelStandard::PreProcess(MParList *pList)
{
    fHil    = (MHillasExt*)pList->FindObject(fHilName, "MHillasExt");
    if (!fHil)
    {
      *fLog << dbginf << "MHillasExt object " << fHilName << " not found... aborting." << endl;
      return kFALSE;
    }

    fHilSrc = (MHillasSrc*)pList->FindObject(fHilSrcName, "MHillasSrc");
    if (!fHilSrc)
    {
      *fLog << dbginf << "MHillasSrc object " << fHilSrcName << " not found... aborting." << endl;
      return kFALSE;
    }


    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    fEvt = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fEvt)
    {
        *fLog << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }


    fCam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fCam)
    {
        *fLog << dbginf << "MGeomCam (Camera Geometry) missing in Parameter List... aborting." << endl;
        return kFALSE;
    }
    fMm2Deg = fCam->GetConvMm2Deg();

    //*fLog << "fMm2Deg = " << fMm2Deg << endl;

    memset(fCut, 0, sizeof(fCut));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Evaluate standard cuts
// 
// if selections are fulfilled set fResult = kTRUE;
// 
//
Bool_t MFCT1SelStandard::Process()
{
    Int_t rc = 0;
    fResult  = kFALSE;

    Double_t length       = fHil->GetLength() * fMm2Deg;
    Double_t width        = fHil->GetWidth()  * fMm2Deg;
    Double_t dist         = fHilSrc->GetDist()* fMm2Deg;
    //Double_t delta        = fHil->GetDelta()  * kRad2Deg;
    Double_t size         = fHil->GetSize();
    Int_t numusedpixels   = fHil->GetNumUsedPixels();
    Int_t numcorepixels   = fHil->GetNumCorePixels();

    if ( numusedpixels >= fUsedPixelsMax  ||  numcorepixels <= fCorePixelsMin )
    {
      rc = 1;
      fResult = kTRUE;
    }    

    else if ( size <= fSizeMin )
    {
      rc = 2;
      fResult = kTRUE;
    }    

    else if ( dist< fDistMin   ||  dist > fDistMax )
    {
      rc = 3;
      fResult = kTRUE;
    }    

    else if ( length <= fLengthMin   ||  width <= fWidthMin )
    {
      rc = 4;
      fResult = kTRUE;
    }    

    fCut[rc]++;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the Standard selections.
//
Bool_t MFCT1SelStandard::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fCut[1] << " (" << setw(3) 
          << (int)(fCut[1]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: Used pixels >= " << fUsedPixelsMax 
          << " or Core pixels <= " << fCorePixelsMin << endl;

    *fLog << " " << setw(7) << fCut[2] << " (" << setw(3) 
          << (int)(fCut[2]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: SIZE <= " << fSizeMin << endl;

    *fLog << " " << setw(7) << fCut[3] << " (" << setw(3) 
          << (int)(fCut[3]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: DIST < " << fDistMin 
          << " or DIST > " << fDistMax << endl;

    *fLog << " " << setw(7) << fCut[4] << " (" << setw(3) 
          << (int)(fCut[4]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: LENGTH <= " << fLengthMin 
          << " or WIDTH <= " << fWidthMin << endl;

    *fLog << " " << fCut[0] << " (" 
          << (int)(fCut[0]*100/GetNumExecutions()) 
          << "%) Evts survived Standard selections!" << endl;
    *fLog << endl;

    return kTRUE;
}
