/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHAlphaEnergyTheta                                                      //
//                                                                          //
//  3D-histogram in alpha, E-est and Theta                                  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHAlphaEnergyTheta.h"

#include <TCanvas.h>

#include <math.h>

#include "MMcEvt.hxx"
#include "MHillasSrc.h"
#include "MEnergyEst.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHAlphaEnergyTheta);

// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram. 
//
MHAlphaEnergyTheta::MHAlphaEnergyTheta(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHAlphaEnergyTheta";
    fTitle = title ? title : "3-D histogram in alpha, energy and theta";

    fHist.SetDirectory(NULL);

    fHist.SetTitle("3D-plot of alpha, E_{est}, Theta");
    fHist.SetXTitle("\\alpha [\\circ]");
    fHist.SetYTitle("E_{est} [GeV]");
    fHist.SetZTitle("\\Theta [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set binnings and prepare filling of the histogram
// 
Bool_t MHAlphaEnergyTheta::SetupFill(const MParList *plist)
{
   fEnergy = (MEnergyEst*)plist->FindObject("MEnergyEst");
   if (!fEnergy)
   {
       *fLog << err << dbginf << "MEnergyEst not found... aborting." << endl;
       return kFALSE;
   }

   fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
   if (!fMcEvt)
   {
       *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
       return kFALSE;
   }

   MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
   MBinning* binsalphaflux  = (MBinning*)plist->FindObject("BinningAlphaFlux");
   MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta");
   if (!binsenergy || !binsalphaflux || !binstheta)
   {
       *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsalphaflux, binsenergy, binstheta);

   fHist.Sumw2(); 

   return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histogram
// 
Bool_t MHAlphaEnergyTheta::Fill(const MParContainer *par, const Stat_t w)
{
    MHillasSrc &hil = *(MHillasSrc*)par;

    fHist.Fill(fabs(hil.GetAlpha()), fEnergy->GetEnergy(),
               fMcEvt->GetTelescopeTheta()*kRad2Deg, w);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
// 
void MHAlphaEnergyTheta::Draw(Option_t *opt)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2,2);

    TH1 *h;

    pad->cd(1);
    gPad->SetBorderMode(0);
    h = fHist.Project3D("expro");
    h->SetTitle("Distribution of \\alpha [\\circ]");
    h->SetXTitle("\\alpha [\\circ]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    pad->cd(2);
    gPad->SetBorderMode(0);
    gPad->SetLogx();
    h = fHist.Project3D("eypro");
    h->SetTitle("Distribution of E-est [GeV]");
    h->SetXTitle("E_{est} [GeV]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    pad->cd(3);
    gPad->SetBorderMode(0);
    h = fHist.Project3D("ezpro");
    h->SetTitle("Distribution of \\Theta [\\circ]");
    h->SetXTitle("\\Theta [\\circ]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    pad->cd(4);
    gPad->SetBorderMode(0);
    fHist.Draw(opt);

    pad->Modified();
    pad->Update();
}
