#ifndef MARS_MCamDisplay
#define MARS_MCamDisplay

#ifndef MARS_MAGIC
#include "MAGIC.h"
#endif
#ifndef ROOT_Gtypes
#include <Gtypes.h>
#endif
#ifndef ROOT_TArrayF
#include <TArrayF.h>
#endif
#ifndef ROOT_TArrayI
#include <TArrayI.h>
#endif
#ifndef ROOT_TClonesArray
#include <TClonesArray.h>
#endif


class TBox;
class TText;
class TArrow;
class TGStatusBar;

class MGeomCam;
class MHexagon;
class MCurrents;
class MCamEvent;
class MRflEvtData;
class MCerPhotEvt;
class MCerPhotPix;
class MImgCleanStd;
class MPedestalPix;
class MPedestalCam;

class MCamDisplay : public TObject
{
private:
    MGeomCam      *fGeomCam;     // pointer to camera geometry

    Bool_t         fAutoScale;   // indicating the autoscale function

    UInt_t         fNumPixels;   // number of pixels in the present geometry
    Float_t        fRange;       // the range in millimeters of the present geometry

    TArrayI        fColors;      // Color conversion table

    TArrow        *fArrowX;      // Coordinate System
    TArrow        *fArrowY;      // Coordinate System

    TText         *fLegRadius;   // Coordinate System
    TText         *fLegDegree;   // Coordinate System

    TClonesArray  *fPixels;      // array of all hexagons
    TClonesArray  *fLegend;      // array of all color bars
    TClonesArray  *fLegText;     // array of all texts
    TClonesArray  *fPhotons;     // array of reflector photons
 
    TArrayF        fData;        //
    Float_t        fMinimum;
    Float_t        fMaximum;

    TList         *fNotify;      //!

    //TGStatusBar   *fStatusBar;

    TBox  *GetBox(Int_t i)  { return (TBox*) fLegend->At(i); }

    MHexagon &operator[](int i) { return *((MHexagon*)fPixels->At(i)); }

    Int_t GetColor(Float_t val, Float_t min, Float_t max, Bool_t islog);

    void  Update(Bool_t islog);
    void  UpdateLegend(Float_t min, Float_t max, Bool_t islog);
    void  SetRange();
    void  SetPalette();

    Int_t GetPixelIndex(Int_t px, Int_t py) const;

    enum {
        kIsUsed = BIT(14)
    };

public:
    MCamDisplay();
    MCamDisplay(MGeomCam *geom);
    ~MCamDisplay();

    void  SetAutoScale(Bool_t input=kTRUE); // *MENU*

    void  ShowRflEvent(const MRflEvtData *event=NULL, EMarkerStyle m=kFullDotMedium);
    void  FillRflEvent(const MRflEvtData &event);
    void  FillLevels(const MCerPhotEvt &event, Float_t lvl1, Float_t lvl2);
    void  FillLevels(const MCerPhotEvt &event, const MImgCleanStd &clean);
    void  Fill(const MCamEvent &event, Int_t type=0);
    void  Fill(const TArrayF &event, Bool_t ispos=kTRUE);
    void  FillRandom();

    void  DrawPixelNumbers();

    void  Paint(Option_t *option="");
    void  Reset();
    void  Draw(Option_t *option="");
    void  SavePrimitive(ofstream &out, Option_t *);
    Int_t DistancetoPrimitive(Int_t px, Int_t py);
    char *GetObjectInfo(Int_t px, Int_t py) const;
    void  ExecuteEvent(Int_t event, Int_t px, Int_t py);

    void  SetPalette(Int_t ncolors, Int_t *colors);

    void  SetPrettyPalette(); // *MENU*
    void  SetDeepBlueSeaPalette(); // *MENU*
    void  SetInvDeepBlueSeaPalette(); // *MENU*

    void  SetPix(const Int_t pixnum, const Int_t color, Float_t min, Float_t max);      // New function added by M.Lopez in 31-01-03

    void  SetMinimum(Float_t m); // *MENU*
    void  SetMaximum(Float_t m); // *MENU*

    void  AddNotify(const MCamEvent &event) { fNotify->Add((TObject*)(&event)); }

    //void SetStatusBar(TGStatusBar *bar) { fStatusBar = bar; }

    ClassDef(MCamDisplay, 1) // Displays the magic camera
};

inline void MCamDisplay::SetAutoScale(Bool_t input)
{
    fAutoScale = input;
}

inline void MCamDisplay::SetMinimum(Float_t m)
{
    fMinimum = m;
}

inline void MCamDisplay::SetMaximum(Float_t m)
{
    fMaximum = m;
}

#endif
