#include <TSystem.h>

#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MRawFileRead.h"
#include "MRawFileWrite.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MArgs.h"
#include "MTime.h"
#include "MArray.h"
#include "MRawEvtData.h"
#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"
#include "MRawCrateArray.h"

using namespace std;

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// This is an easy implementation of the Merging process                    //
// (as compilable prog)                                                     //
//                                                                          //
// at the moment it reads a binary file ("rawtest.bin") which was written   //
// in the DAQ raw format.                                                   //
//                                                                          //
// The data are stored in root container objects (classes derived from      //
// TObject like MRawRunHeader)                                              //
//                                                                          //
// This containers are written to a root file ("rawtest.root")              //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

void Usage()
{
    gLog << "Sorry the usage is:" << endl;
    gLog << "   merpp [-vn] inputfile outputfile [compression level]" << endl << endl;
    gLog << "     input file:   Magic DAQ binary file." << endl;
    gLog << "     ouput file:   Merpped root file." << endl;
    gLog << "     compr. level: 1..9 [default=1]" << endl;
    gLog << "     -vn: Verbosity level n [default=2]" << endl << endl;
}

int main(const int argc, const char **argv)
{
    MArgs arg(argc, argv);
    arg.Print();

    gLog << "==================================================" << endl ;
    gLog << "                   MERPP v0.1" << endl;
    gLog << "      MARS Merging and Preprocessing Program" << endl ;
    gLog << "            Compiled on <" << __DATE__ << ">" << endl ;
    gLog << "               Using ROOT v" << ROOTVER << endl ;
    gLog << "==================================================" << endl ;
    gLog << endl;

    //
    // check for the right usage of the program
    //
    if (arg.GetNumArguments()<2 || arg.GetNumArguments()>3)
    {
        Usage();
        return -1;
    }

    //
    // Set verbosity to highest level.
    //
    gLog.SetDebugLevel(arg.HasOption("-v") ? arg.GetInt("-v") : 2);

    //
    // This is to make argv[i] more readable insidethe code
    //
    const char *kNamein   = arg.GetArgumentStr(0);
    const char *kNameout  = arg.GetArgumentStr(1);
    const int   kComprlvl = arg.GetNumArguments()==3 ? arg.GetArgumentInt(2) : 1;

    //
    //     initialize ROOT  (this is obligatory here)
    //
    TROOT merpp("merpp", "Mars - Merging and Preprocessing Program");
    merpp.SetBatch();

    //
    // check whether the given files are OK.
    //
    if (gSystem->AccessPathName(kNamein, kFileExists))
    {
        gLog << err << "Sorry, the input file '" << kNamein << "' doesn't exist." << endl;
        return -1;
    }

    if (!gSystem->AccessPathName(kNameout, kFileExists))
        gLog << warn << "Warning: A file '" << kNameout << "' exists." << endl;
    else
        if (!gSystem->AccessPathName(kNameout, kWritePermission))
        {
            gLog << err << "Sorry, you don't have write permission for '" << kNameout << "'." << endl;
            return -1;
        }

    MArray::Class()->IgnoreTObjectStreamer();
    MParContainer::Class()->IgnoreTObjectStreamer();

    //
    // create a (empty) list of parameters which can be used by the tasks
    // and an (empty) list of tasks which should be executed
    //
    MParList plist;

    MTaskList tasks;
    plist.AddToList(&tasks);

    //
    // ---- Tho following is only necessary to supress some output ----
    //
    MRawRunHeader runheader;
    plist.AddToList(&runheader);

    MRawEvtHeader evtheader;
    plist.AddToList(&evtheader);

    MRawEvtData evtdata;
    plist.AddToList(&evtdata);

    MRawCrateArray cratearray;
    plist.AddToList(&cratearray);

    MTime evttime("MRawEvtTime");
    plist.AddToList(&evttime);

    //
    // create the tasks which should be executed and add them to the list
    // in the case you don't need parameter containers, all of them can
    // be created by MRawFileRead::PreProcess
    //
    MRawFileRead  reader(kNamein);
    MRawFileWrite writer(kNameout, "RECREATE", "Magic root-file", kComprlvl);
    tasks.AddToList(&reader);
    tasks.AddToList(&writer);

    //
    // create the looping object and tell it about the parameters to use
    // and the tasks to execute
    //
    MEvtLoop magic;
    magic.SetParList(&plist);

    //
    // Start the eventloop which reads the raw file (MRawFileRead) and
    // write all the information into a root file (MRawFileWrite)
    //
    // between reading and writing we can do, transformations, checks, etc.
    // (I'm think of a task like MRawDataCheck)
    //
    if (!magic.Eventloop())
    {
        gLog << err << "ERROR: Merging and preprocessing failed!" << endl;
        return -1;
    }

    gLog << all << "Merpp finished successfull!" << endl;
    return 0;
}


