/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

#include "MDataCheck.h"

#include <TGButton.h>  // TGTextButton

#include "MGList.h"
#include "MFillH.h"
#include "MParList.h"
#include "MEvtLoop.h"
#include "MHFadcCam.h"
#include "MTaskList.h"
#include "MReadTree.h"
#include "MGDisplayAdc.h"

// ---

ClassImp(MDataCheck)

enum {
    kButPedAdc = 0x100,
    kButEvtAdc = 0x101,
    kButPedTdc = 0x102,
    kButEvtTdc = 0x103
};

// --------------------------------------------------------------------------
//
//  Create the 'Data Check' GUI Controls (Window) on the screen. To use it
//  from within the interpreter you can call a Standard Version with
//  'new MDataCheck()'
//
MDataCheck::MDataCheck(/*const TGWindow *main,*/ const TGWindow *p,
                       const UInt_t w, const UInt_t h)
: MBrowser(/*main,*/ p, w, h)
{
    //TGTextButton *pedadc = new TGTextButton(fTop1, "ADC Spectra of Pedestals", kButPedAdc);
    TGTextButton *cradc  = new TGTextButton(fTop1, "ADC Spectra of Cosmics", kButEvtAdc);
    TGTextButton *evttdc = new TGTextButton(fTop1, "Time Spectra of Cosmics", kButEvtTdc);
    //TGTextButton *crtdc  = new TGTextButton(fTop3, "Run Spectra of Cosmics",    kButEvtTdc);

    //pedadc->Associate(this);
    cradc ->Associate(this);
    evttdc->Associate(this);
    //crtdc ->Associate(this);

    //fList->Add(pedadc);
    fList->Add(cradc);
    fList->Add(evttdc);
    //fList->Add(crtdc);

    TGLayoutHints *laybut = new TGLayoutHints(kLHintsTop|kLHintsLeft, 10, 10, 5, 5);
    fList->Add(laybut);

    //fTop1->AddFrame(pedadc, laybut);
    fTop1->AddFrame(cradc,  laybut);
    fTop1->AddFrame(evttdc,  laybut);
    //fTop3->AddFrame(pedtdc, laybut);
    //fTop3->AddFrame(crtdc,  laybut);

    MapSubwindows();

    Layout();

    SetWindowName("DataCheck Window");
    SetIconName("DataCheck");

    MapWindow();
}

// --------------------------------------------------------------------------
//
//  Create the 'View Adc' GUI Controls (Window) on the screen.
//  Therefor we have to process all data in a file and fill the corresponding
//  histograms.
//
void MDataCheck::View(const char *inputfile, const char *treeName, MHFadcPix::Type_t t)
{
    //
    // create a (empty) list of parameters which can be used by the tasks
    // and an (empty) list of tasks which should be executed
    // connect them in the required way.
    //

    //
    //   create the data containers for the raw data
    //
    MParList plist;

    //
    // set up the tasks for this job
    //
    MTaskList tasks;
    plist.AddToList(&tasks);

    MReadTree read(treeName, inputfile);
    read.DisableAutoScheme();

    MHFadcCam hist(577, t);
    plist.AddToList(&hist);

    MFillH fill(&hist, "MRawEvtData");

    tasks.AddToList(&read);
    tasks.AddToList(&fill);

    //
    // set up the loop for the processing
    //
    MEvtLoop magic;
    magic.SetParList(&plist);

    //
    // Add ProgressBar to window
    //
    TGProgressBar *bar = CreateProgressBar(fTop1);
    magic.SetProgressBar(bar);

    //
    // Execute your analysis
    //
    Bool_t rc = magic.Eventloop();

    //
    // Remove progressbar from window
    //
    DestroyProgressBar(bar);

    if (!rc)
        return;

    new MGDisplayAdc(&hist);
}

// --------------------------------------------------------------------------
//
//  Process the GUI control events (here: mainly button clicks)
//
Bool_t MDataCheck::ProcessMessage(Long_t msg, Long_t parm1, Long_t parm2)
{
    // Process events generated by the buttons in the frame.

    if (GET_MSG(msg)!=kC_COMMAND || GET_SUBMSG(msg)!=kCM_BUTTON)
        return MBrowser::ProcessMessage(msg, parm1, parm2);

    switch (parm1)
    {
    case kButPedAdc:
    case kButEvtAdc:
    case kButPedTdc:
    case kButEvtTdc:
        if (!InputFileSelected())
        {
            DisplError("No Input (root) File selected!");
            return kTRUE;
        }

        switch (parm1)
        {
        case kButPedAdc:
            View(fInputFile, "PedEvents", MHFadcPix::kValue);
            return kTRUE;

        case kButEvtAdc:
            View(fInputFile, "Events", MHFadcPix::kValue);
            return kTRUE;

        case kButPedTdc:
            View(fInputFile, "PedEvents", MHFadcPix::kSlices);
            return kTRUE;

        case kButEvtTdc:
            View(fInputFile, "Events", MHFadcPix::kSlices);
            return kTRUE;
        }
        return kTRUE;
    }

    return MBrowser::ProcessMessage(msg, parm1, parm2);
}
