//=//////////////////////////////////////////////////////////////////////
//=
//= creadparam                
//=
//= @file        creadparam.h
//= @desc        Header file
//= @author      J C Gonzalez
//= @email       gonzalez@mppmu.mpg.de
//= @date        Thu May  7 16:24:22 1998
//=
//=----------------------------------------------------------------------
//=
//= Created: Thu May  7 16:24:22 1998
//= Author:  Jose Carlos Gonzalez
//= Purpose: Program for reflector simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: creadparam.h,v $
//= $Revision: 1.18 $
//= $Author: blanch $ 
//= $Date: 2003-09-15 10:23:31 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |creadparam.h|.

/*!@"

  In this section you can find the source code for the file
  |creadparam.h|.  This file is mainly needed by
  |creadparam.cxx|. Here is located the definition of the commands you
  can use in the parameters file. In this file, the first line must be
  |camera 'version'|, where |'version'| is the appropiate version of
  the output format (NOT the version of the camera program) which can
  read the commands written in that parameters file. You cannot mix
  parameters files and executables with different versions. The
  current version is |camera 0.2|.

  The commands now available for the parameters file are:

  @itemize
  
  @- |input_file| filename :    
     Sets the name of the input file (|.rfl|).
  @- |starfield_file| filename :    
     Sets the name of the starfield input file (|.rfl|).
  @- |ct_file| filename :    
     Sets the name of the CT definition file (|.def|).
  @- |data_file| filename :    
     Sets the name of the output data file (|.dat|).
  @- |elec_noise_off| :    
     De-activates ElecNoise for FADC and Trigger channels. By default it is on.
  @- |fadc_noise_off| :    
     Sets the  ElecNoise for FADC.
  @- |trig_noise_off| :    
     Sets the  ElecNoise for trigger system.
  @- |nsb_on| :    
     Activates the NSB simulation. This is the default.
  @- |nsb_off| :    
     De-activates the NSB simulation.
  @- |nsb_mean| number :    
     Sets the mean value for the NSB.
     Default value: 6 for CT1, 6 for MAGIC.
     This implies always |nsb_on|.
  @- |threshold| number :    
     Sets the Threshold value q0. Default value: 10.
  @- |tail_cut| number : 
     Sets the Tail-Cut value.
     Default value: 7.
  @- |islands_cut| number :    
     Sets the Islands-Cut value i0.
     Default value: 10.
  @- |end_file|
     Last command in the parameters file.

  @enditemize

  @ignoreHTML
  A parameters file (a small one) looks like this:

  |camera 0.2|

  |input_file    gm100-500.rfl|

  |output_file   gm100-500.phe|

  |output_file   gm100-500.dat|

  |ct_file       magic.def|

  |threshold     10.0|

  |tail_cut      7.0|

  |nsb_mean      5.0|

  |end_file|
  @endignoreHTML

@"*/

//!@{

#ifndef _creadparam_
#define _creadparam_

#ifndef _this_
#define _this_ creadparam
#endif

//!@}

//!@subsection Include files.

//!@{

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>

#include "jcmacros.h"
#include "jcdebug.h"

#include "camera-v.h"

#include "MTriggerDefine.h"
#include "MFadcDefine.h"
#include "Mdefine.h"

//!@}

//!@subsection Macro-definitions, and constants.

//!@{

// now we define the list PM_ITEM_LIST of possible items in
// the parameters file. note that they are functions of
// T(x). we will change T(x) to be the real item or the
// string of this item when needed

#define PM_ITEM_LIST   /* LIST OF ITEMS IN THE PARAMETERS FILE */  \
T(input_file),      /* input file */                              \
T(starfield_file),  /* starfield file */                          \
T(starfield_center),/* center of thestarfield FOV */              \
T(data_file),       /* data file */                               \
T(root_file),       /* ROOT file */                               \
T(ct_file),         /* file with the characteristics of the CT */ \
T(ct_num),          /* number of CT, which is number of reflector files */ \
T(ct_geom),         /* class with the geometry of the CT camera */ \
T(qe_file),         /* file with the characteristics of the CT */ \
T(ana_pixels),      /* size of the camera for parameters calculation */  \
T(elec_noise_off),  /* de-activates ElecNoise for FADC and Trigger channels */\
T(fadc_pedestal),      /* Pedestal value for FADC */\
T(fadc_noise),      /* ElecNoise for FADC */\
T(trig_noise),      /* ElecNoise for trigger */\
T(sfr_on),          /* starfield rotation */  \
T(nsb_on),          /* activates NSB simulation */ \
T(nsb_off),         /* de-activates NSB simulation */ \
T(nsb_mean),        /* mean value of NSB contribution per pixel */ \
T(nsb_directory),   /* database for the NSB  */ \
T(nsb_dir_outer),   /* database for the NSB  */ \
T(pixel_thres),     /* value of q0 for trigger for a given pixel */ \
T(secure_disc),     /* Ac value to rise discrimnator threshold */ \
T(seeds),           /* seeds for random number generation */ \
T(data_from_stdin), /* to read data from STDIN */ \
T(skip),            /* skip pathological showers */ \
T(write_all_events),/* write to file .phe ALL images (even w.o. trigger)*/ \
T(nowrite_McEvt),   /* nowrite to file .root McEvt infos */ \
T(write_McTrig),    /* write to file .root McTrig infos */ \
T(write_McFadc),    /* write to file .root McFadc infos */ \
T(nowrite_RawEvt),   /* write to file .root RawData infos */ \
T(write_all_data),  /* write to file .dat ALL image data */ \
T(select_energy),   /* energy range to read: only for .phe files */ \
T(trigger_scan),    /* show signal in the diskriminator */ \
T(fadc_scan),       /* show fadc signal */ \
T(trigger_loop),    /* make loop over trigger conditions */ \
T(trigger_prop),    /* values of trigger properties */ \
T(fadc_prop),    /* values of fadc properties */ \
T(fadc_outer),    /* values of fadc properties for outer pixels */ \
T(trigger_single),  /* trigger conditions */ \
T(Trigger_Loop_Output_Only_Specialists),   /* special steercard */ \
T(source_offset),   /* Displacement of the source from the center */ \
T(end_file)         /* end of the parameters file */
  
#define T(x)  x             // define T() as the name as it is

enum ITEM_TYPE {
  PM_ITEM_LIST
};

#undef T

#define T(x)  #x              // define T() as the string of x

const char *const ITEM_NAMES[] = {
  PM_ITEM_LIST
};

#undef T

#define LINE_MAX_LENGTH  400
#define ITEM_MAX_LENGTH  40
#define PATH_MAX_LENGTH  120

//!@}

//!@subsection Prototypes of functions.

//!@{

//++
// prototypes
//--

void readparam(char * filename);
char *get_input_filename(int i);
char *get_starfield_filename(void);
void get_starfield_center(int *rh,int *rm,int *rs,int *dd,int *dm,int *ds);
int get_starfield_rotate(void);
char *get_data_filename(void);
char *get_root_filename(void);
char *get_loop_filename(void);
char *get_ct_filename(void);
int get_ct_number(void);
int get_ct_geometry(void);
char *get_qe_filename(int ict=0);
char *get_nsb_directory(void);
char *get_nsb_directory_outer(void);
int add_elec_noise(float *fadc, float *trig);
int get_nsb(float *n, int *m);
long int get_seeds(int n);
int get_ana_pixels(void);
void get_skip_showers( int *s ); 
int get_nskip_showers( void ); 
int get_data_from_stdin(void);
int get_write_all_events(void);
int get_write_McEvt(void);
int get_write_McTrig(void);
int get_write_McFadc(void);
int get_write_RawEvt(void);
int get_write_all_data(void);
int get_select_energy(float *le, float *ue);
int get_FADC_Scan(void);
int get_Trigger_Scan(void);
void get_Trigger_properties(float *gl, float *ot, float *ra, float *rf);
void get_FADC_properties(float *ra, float *rf, float *rao, float *rfo);
float get_FADC_pedestal(void);
int get_Trigger_Loop(float *lt, float *ut, float *st, int *lm, int *um, int *lg, int *ug);
void get_Trigger_Single(float **t, int *m, int *g);
void get_threshold(float *t);
void get_secure_threhold(float *ac, float *disc);
int get_indi_thres_pixel(void);
void get_source_off(float *th, float *p);
//!@}

//!@{

#endif // ! _creadparam_

//!@}

//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{

/*
 * $Log: not supported by cvs2svn $
 * Revision 1.17  2003/07/17 18:05:13  blanch
 * Header file for creadparam.cxx 1.20
 *
 * Revision 1.16  2003/01/14 13:37:32  blanch
 * Option to set a dc value to rise the discriminator threshold has been added.
 *
 * Revision 1.15  2002/07/16 16:20:59  blanch
 * Modifications done for the camera.cxx version, where a first implementation
 * of the Star Field Rotation has been introduced.
 *
 * Revision 1.14  2002/03/15 15:17:57  blanch
 * Header file for creadparam.cxx v 1.16
 *
 * Revision 1.13  2002/03/04 17:20:50  blanch
 * Header file for creadparam.cxx 1.15
 *
 * Revision 1.12  2002/02/28 15:08:01  blanch
 * Header file for creadparam.cxx 1.14.
 *
 * Revision 1.11  2002/01/18 17:45:37  blanch
 * Header file for creadparam.cxx 1.13.
 *
 * Revision 1.10  2001/11/13 17:06:01  blanch
 * Header file for creadparam.cxx version 1.12
 *
 * Revision 1.9  2001/07/19 09:29:02  blanch
 * Header file for creadparam.cxx vrsion 1.10
 *
 * Revision 1.8  2001/03/05 10:44:44  blanch
 * Header file of creadparam.cxx version 1.9
 *
 * Revision 1.7  2001/02/23 10:56:10  magicsol
 * Header file for creadparam.cxx version 1.8
 *
 * Revision 1.6  2001/01/15 12:38:00  magicsol
 * *** empty log message ***
 *
 * Revision 1.5  2000/07/04 14:13:34  MagicSol
 * *** empty log message ***
 *
 * Revision 1.4  2000/05/11 14:23:28  blanch
 * Thi version owns to the version 1.4 of creadparam.cxx.
 *
 * Revision 1.3  2000/03/24 18:15:46  blanch
 * *** empty log message ***
 *
 * Revision 1.2  2000/02/18 17:50:07  petry
 * This version belongs to camera.cxx 1.5.
 * It version has been put in the repository in order to be
 * able to share the further development with others.
 *
 * If you need something working, wait or take an earlier one.
 * See file README
 *
 * Revision 1.1.1.1  1999/11/05 11:59:34  harald
 * This the starting point for CVS controlled further developments of the
 * camera program. The program was originally written by Jose Carlos. 
 * But here you can find a "rootified" version to the program. This means 
 * that there is no hbook stuff in it now. Also the output of the
 * program changed to the MagicRawDataFormat. 
 *
 * The "rootification" was done by Dirk Petry and Harald Kornmayer. 
 *
 * Revision 1.2  1999/10/22 15:01:29  petry
 * version sent to H.K. and N.M. on Fri Oct 22 1999
 *
 * Revision 1.1.1.1  1999/10/21 16:35:11  petry
 * first synthesised version
 *
 * Revision 1.7  1999/03/15  14:59:09  gonzalez
 * camera-1_1
 *
 * Revision 1.6  1999/03/02  09:56:13  gonzalez
 * *** empty log message ***
 *
 * Revision 1.5  1999/01/14  17:32:43  gonzalez
 * Added to camera the STDIN input option (data_from_input)
 *
 */

//!@}
//=EOF
