#include "Slalib.h"

#include <time.h>
#include <iostream.h>  // cout

#include "coord.h"

#include "slalib.h"

ClassImp(Slalib);

Slalib::Slalib(MObservatory::LocationName_t key) : Timer(), MObservatory(key)
{
}

Slalib::~Slalib()
{
}

Double_t Slalib::Trunc(Double_t val)
{
    /* dint(A) - truncate to nearest whole number towards zero (double) */
    return val<0 ? TMath::Ceil(val) : TMath::Floor(val);
}

Double_t Slalib::Round(Double_t val)
{
    /* dnint(A) - round to nearest whole number (double) */
    return val<0 ? TMath::Ceil(val-0.5) : TMath::Floor(val+0.5);
}

void Slalib::SetMjd(double mjd)
{
    Timer::SetMjd(mjd);
    fAlpha = slaGmst(mjd) + GetElong();
}

ZdAz Slalib::XYZ2ZdAz(double coord[3]) const
{
    //
    // -- xyz to spherical coordinates --
    //
    double ra, dec;
    slaDcc2s(coord, &ra, &dec);

    //
    // radec[rad] -> hadec[rad]
    //
    const double ha = fAlpha-ra;

    //
    // hadec[rad] -> altaz[rad]
    //
    double alt, az;
    slaDe2h(ha, dec, GetPhi(), &az, &alt);

    return ZdAz(kPiDiv2-alt, az);
}

Double_t Slalib::Hms2Sec(Int_t deg, UInt_t min, Double_t sec, Char_t sgn)
{
    const Double_t rc = TMath::Sign((60.0 * (60.0 * (Double_t)TMath::Abs(deg) + (Double_t)min) + sec), (Double_t)deg);
    return sgn=='-' ? -rc : rc;
}

Double_t Slalib::Dms2Rad(Int_t deg, UInt_t min, Double_t sec, Char_t sgn)
{
    /* pi/(180*3600):  arcseconds to radians */
#define DAS2R 4.8481368110953599358991410235794797595635330237270e-6
    return Hms2Sec(deg, min, sec, sgn)*DAS2R;
}

Double_t Slalib::Hms2Rad(Int_t hor, UInt_t min, Double_t sec, Char_t sgn)
{
    /* pi/(12*3600):  seconds of time to radians */
#define DS2R 7.2722052166430399038487115353692196393452995355905e-5
    return Hms2Sec(hor, min, sec, sgn)*DS2R;
}

Double_t Slalib::Dms2Deg(Int_t deg, UInt_t min, Double_t sec, Char_t sgn)
{
    return Hms2Sec(deg, min, sec, sgn)/3600.;
}

Double_t Slalib::Hms2Deg(Int_t hor, UInt_t min, Double_t sec, Char_t sgn)
{
    return Hms2Sec(hor, min, sec, sgn)/240.;
}

Double_t Slalib::Dms2Hor(Int_t deg, UInt_t min, Double_t sec, Char_t sgn)
{
    return Hms2Sec(deg, min, sec, sgn)/15.;
}

Double_t Slalib::Hms2Hor(Int_t hor, UInt_t min, Double_t sec, Char_t sgn)
{
    return Hms2Sec(hor, min, sec, sgn)/3600.;
}

void Slalib::Day2Hms(Double_t day, Char_t &sgn, UShort_t &hor, UShort_t &min, UShort_t &sec)
{
    /* Handle sign */
    sgn = day<0?'-':'+';

    /* Round interval and express in smallest units required */
    Double_t a = Round(86400. * TMath::Abs(day)); // Days to seconds

    /* Separate into fields */
    const Double_t ah = Trunc(a/3600.);
    a -= ah * 3600.;
    const Double_t am = Trunc(a/60.);
    a -= am * 60.;
    const Double_t as = Trunc(a);

    /* Return results */
    hor = (UShort_t)ah;
    min = (UShort_t)am;
    sec = (UShort_t)as;
}

void Slalib::Rad2Hms(Double_t rad, Char_t &sgn, UShort_t &deg, UShort_t &min, UShort_t &sec)
{
    Day2Hms(rad/(TMath::Pi()*2), sgn, deg, min, sec);
}

void Slalib::Rad2Dms(Double_t rad, Char_t &sgn, UShort_t &deg, UShort_t &min, UShort_t &sec)
{
    Rad2Hms(rad*15, sgn, deg, min, sec);
}

void Slalib::Deg2Dms(Double_t d, Char_t &sgn, UShort_t &deg, UShort_t &min, UShort_t &sec)
{
    Day2Hms(d/24, sgn, deg, min, sec);
}

void Slalib::Deg2Hms(Double_t d, Char_t &sgn, UShort_t &deg, UShort_t &min, UShort_t &sec)
{
    Rad2Hms(d/360, sgn, deg, min, sec);
}

void Slalib::Hor2Dms(Double_t h, Char_t &sgn, UShort_t &deg, UShort_t &min, UShort_t &sec)
{
    Day2Hms(h*15/24, sgn, deg, min, sec);
}

void Slalib::Hor2Hms(Double_t h, Char_t &sgn, UShort_t &deg, UShort_t &min, UShort_t &sec)
{
    Day2Hms(h/24, sgn, deg, min, sec);
}

void Slalib::Day2Hm(Double_t day, Char_t &sgn, UShort_t &hor, Double_t &min)
{
    /* Handle sign */
    sgn = day<0?'-':'+';

    /* Round interval and express in smallest units required */
    Double_t a = Round(86400. * TMath::Abs(day)); // Days to seconds

    /* Separate into fields */
    const Double_t ah = Trunc(a/3600.);
    a -= ah * 3600.;

    /* Return results */
    hor = (UShort_t)ah;
    min = a/60.;
}

void Slalib::Rad2Hm(Double_t rad, Char_t &sgn, UShort_t &deg, Double_t &min)
{
    Day2Hm(rad/(TMath::Pi()*2), sgn, deg, min);
}

void Slalib::Rad2Dm(Double_t rad, Char_t &sgn, UShort_t &deg, Double_t &min)
{
    Rad2Hm(rad*15, sgn, deg, min);
}

void Slalib::Deg2Dm(Double_t d, Char_t &sgn, UShort_t &deg, Double_t &min)
{
    Day2Hm(d/24, sgn, deg, min);
}

void Slalib::Deg2Hm(Double_t d, Char_t &sgn, UShort_t &deg, Double_t &min)
{
    Rad2Hm(d/360, sgn, deg, min);
}

void Slalib::Hor2Dm(Double_t h, Char_t &sgn, UShort_t &deg, Double_t &min)
{
    Day2Hm(h*15/24, sgn, deg, min);
}

void Slalib::Hor2Hm(Double_t h, Char_t &sgn, UShort_t &deg, Double_t &min)
{
    Day2Hm(h/24, sgn, deg, min);
}

