#ifndef COSY_MBending
#define COSY_MBending

#include "coord.h"

#ifndef ROOT_TArrayD
#include <TArrayD.h>
#endif

class TMinuit;

class MBending
{
private:
    static const Int_t fNumPar;

    Double_t fIe   ; // [rad] Index Error in Elevation
    Double_t fIa   ; // [rad] Index Error in Azimuth
    Double_t fFlop ; // [rad] Vertical Sag
    Double_t fNpae ; // [rad] Az-El Nonperpendicularity
    Double_t fCa   ; // [rad] Left-Right Collimation Error
    Double_t fAn   ; // [rad] Azimuth Axis Misalignment (N-S)
    Double_t fAw   ; // [rad] Azimuth Axis Misalignment (E-W)
    Double_t fTf   ; // [rad] Tube fluxture (sin)
    Double_t fTx   ; // [rad] Tube fluxture (tan)
    Double_t fNrx  ; // [rad] Nasmyth rotator displacement, horizontal
    Double_t fNry  ; // [rad] Nasmyth rotator displacement, vertical
    Double_t fCrx  ; // [rad] Alt/Az Coude Displacement (N-S)
    Double_t fCry  ; // [rad] Alt/Az Coude Displacement (E-W)
    Double_t fEces ; // [rad] Elevation Centering Error (sin)
    Double_t fAces ; // [rad] Azimuth Centering Error (sin)
    Double_t fEcec ; // [rad] Elevation Centering Error (cos)
    Double_t fAcec ; // [rad] Azimuth Centering Error (cos)
    Double_t fMagic1; // [rad] Magic Term (what is it?)
    Double_t fMagic2; // [rad] Magic Term (what is it?)

    Double_t **fCoeff; //!
    TString   *fName;  //!
    TString   *fDescr; //!

    TArrayD   fError;

    void Init();

    void Clear()
    {
        for (int i=0; i<fNumPar; i++)
        {
            *fCoeff[i] = 0;
            fError[i] = 0;
        }
    }

    static Double_t Sign(Double_t val, Double_t alt);

public:
    MBending() { fError.Set(fNumPar); Init(); Clear(); }
    MBending(const char *name) { fError.Set(fNumPar); Init(); Clear(); Load(name); }
    virtual ~MBending() { delete fName; delete fCoeff; delete fDescr; }

    void Load(const char *name);
    void Save(const char *name);

    void Reset();

    ZdAz  Correct(const ZdAz &zdaz) const;
    AltAz Correct(const AltAz &aaz) const;

    ZdAz  CorrectBack(const ZdAz &zdaz) const;
    AltAz CorrectBack(const AltAz &aaz) const;

    ZdAz operator()(const ZdAz &zdaz) const { return Correct(zdaz); }
    AltAz operator()(const AltAz &aaz) const { return Correct(aaz); }

    ZdAz operator()(const ZdAz &zdaz, void (*fcn)(ZdAz &zdaz, Double_t *par)) const
    {
        Double_t par[fNumPar];
        GetParameters(par);
        ZdAz za = zdaz;
        fcn(za, par);
        return za;
    }

    AltAz operator()(const AltAz &aaz, void (*fcn)(AltAz &aaz, Double_t *par)) const
    {
        Double_t par[fNumPar];
        GetParameters(par);
        AltAz aa = aaz;
        fcn(aa, par);
        return aa;
    }

    void SetParameters(const Double_t *par, Int_t n=fNumPar);
    void GetParameters(Double_t *par, Int_t n=fNumPar) const;

    void SetParameters(const TArrayD &par)
    {
        SetParameters(par.GetArray(), par.GetSize());
    }
    void GetParameters(TArrayD &par) const
    {
        par.Set(fNumPar);
        GetParameters(par.GetArray());
    }
    void GetError(TArrayD &par) const
    {
        par = fError;
        for (int i=0; i<fNumPar; i++)
            par[i] *= kRad2Deg;
    }

    void SetMinuitParameters(TMinuit &m, Int_t n=-1) const;
    void GetMinuitParameters(TMinuit &m, Int_t n=-1);
    void PrintMinuitParameters(TMinuit &m, Int_t n=-1) const;

    const TString &GetName(int i) const { return fName[i]; }
    const TString &GetDescription(int i) const { return fDescr[i]; }

    /*
     Double_t GetIe() const { return fIe; }
     Double_t GetIa() const { return fIa; }
     Double_t GetCa() const { return fCa; }
     Double_t GetAn() const { return fAn; }
     Double_t GetAw() const { return fAw; }
     Double_t GetNrx() const { return fNrx; }
     Double_t GetNry() const { return fNry; }
     Double_t GetCrx() const { return fNrx; }
     Double_t GetCry() const { return fNry; }
     Double_t GetEces() const { return fEces; }
     Double_t GetEcec() const { return fEcec; }
     Double_t GetAces() const { return fAces; }
     Double_t GetAcec() const { return fAcec; }
     Double_t GetNpae() const { return fNpae; }
     */

    static const Int_t GetNumPar() { return fNumPar; }
    ClassDef(MBending, 1)
};

#endif
