/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek, 10/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MH2dimFunction
//
// container of histograms for the parameters of the 2-dim function
// describing the shower image
//
////////////////////////////////////////////////////////////////////////////
#include "MH2dimFunction.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"
#include "MBinning.h"
#include "MParList.h"

#include "MHillas.h"
#include "M2dimFunction.h"

ClassImp(MH2dimFunction);

using namespace std;

// --------------------------------------------------------------------------
//
// Setup histograms 
//
MH2dimFunction::MH2dimFunction(const char *name, const char *title)
{
    fName  = name  ? name  : "MH2dimFunction";
    fTitle = title ? title : "Histograms of parameters of 2-dim function";

    fHistXbarYbar.SetName("shower-maximum");
    fHistXbarYbar.SetTitle("position of maximum of shower");
    fHistXbarYbar.SetXTitle("x-position of maximum of shower image");
    fHistXbarYbar.SetYTitle("y-position of maximum of shower image");
    fHistXbarYbar.SetDirectory(NULL);
    fHistXbarYbar.SetFillStyle(4000);
    fHistXbarYbar.UseCurrentStyle();

    fHistAmp.SetName("Amplitude parameter");
    fHistAmp.SetTitle("amplitude parameter");
    fHistAmp.SetXTitle("amplitude parameter");
    fHistAmp.SetYTitle("Counts");
    fHistAmp.SetDirectory(NULL);
    fHistAmp.SetFillStyle(4000);
    fHistAmp.UseCurrentStyle();

    fHistMajor.SetName("Length parameter");
    fHistMajor.SetTitle("length parameter");
    fHistMajor.SetXTitle("length parameter");
    fHistMajor.SetYTitle("Counts");
    fHistMajor.SetDirectory(NULL);
    fHistMajor.SetLineColor(kBlue);
    fHistMajor.SetFillStyle(4000);
    fHistMajor.UseCurrentStyle();

    fHistMinor.SetName("Width parameter");
    fHistMinor.SetTitle("width parameter");
    fHistMinor.SetXTitle("width parameter");
    fHistMinor.SetYTitle("Counts");
    fHistMinor.SetDirectory(NULL);
    fHistMinor.SetLineColor(kGreen);
    fHistMinor.SetFillStyle(4000);
    fHistMinor.UseCurrentStyle();

    fHistAsym.SetName("Asymmetry parameter");
    fHistAsym.SetTitle("asymmetry parameter");
    fHistAsym.SetXTitle("asymmetry parameter");
    fHistAsym.SetYTitle("Counts");
    fHistAsym.SetDirectory(NULL);
    fHistAsym.SetFillStyle(4000);
    fHistAsym.UseCurrentStyle();
 
    MBinning bins;

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHistAmp);

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHistMajor);

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHistMinor);

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHistAsym);
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning
//
Bool_t MH2dimFunction::SetupFill(const MParList *plist)
{
    ApplyBinning(*plist, "XbarYbar", &fHistXbarYbar);
    ApplyBinning(*plist, "Amp",      &fHistAmp);
    ApplyBinning(*plist, "Major",    &fHistMajor);
    ApplyBinning(*plist, "Minor",    &fHistMinor);
    ApplyBinning(*plist, "Asym",     &fHistAsym);

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Fill the histograms with data from a M2dimFunction container.
//
Bool_t MH2dimFunction::Fill(const MParContainer *par, const Stat_t w)
{
    if (!par)
    {
        *fLog << err << "MH2dimFunction::Fill: Pointer (!=NULL) expected." << endl;
        return kFALSE;
    }

    const M2dimFunction &h = *(M2dimFunction*)par;

    fHistXbarYbar.Fill(h.GetXbar(), h.GetYbar(), w);
    fHistAmp.Fill(h.GetAmp(), w);

    fHistMajor.Fill(h.GetMajor(), w);
    fHistMinor.Fill(h.GetMinor(), w);

    fHistAsym.Fill(h.GetAsym(), w);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MH2dimFunction::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2,2);

    pad->cd(1);
    gPad->SetBorderMode(0);
    TAxis &x = *fHistLeakage1.GetXaxis();
    x.SetRangeUser(0.0, x.GetXmax());
    MH::DrawSame(fHistLeakage1, fHistLeakage2, "Leakage1 and Leakage2");

    pad->cd(2);
    gPad->SetBorderMode(0);
    MH::DrawSame(fHistCorePix, fHistUsedPix, "Number of core/used Pixels");

    pad->cd(3);
    gPad->SetBorderMode(0);
    MH::DrawSame(fHistConc1, fHistConc, "Concentrations");

    pad->cd(4);
    gPad->SetBorderMode(0);

    pad->Modified();
    pad->Update();
}

//==========================================================================






