/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 11/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2002-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MEventRateCalc
//
//
//  Input Containers:
//    MTime
//
//  Output Containers:
//    MEventRate
//
//////////////////////////////////////////////////////////////////////////////
#include "MEventRateCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MTime.h"
#include "MEventRate.h"

ClassImp(MEventRateCalc);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MEventRateCalc::MEventRateCalc(const char *name, const char *title)
    : fNumEvents(10000)
{
    fName  = name  ? name  : "MEventRateCalc";
    fTitle = title ? title : "Calculate trigger rate";

    //AddToBranchList("MRawEvtData.fHiGainPixId");
    //AddToBranchList("MRawEvtData.fLoGainPixId");
    //AddToBranchList("MRawEvtData.fHiGainFadcSamples");
    //AddToBranchList("MRawEvtData.fLoGainFadcSamples");

    //SetDefaultWeights();
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawRunHeader
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//  - MCerPhotEvt
//
Int_t MEventRateCalc::PreProcess(MParList *pList)
{
    fTime = (MTime*)pList->FindObject("MTime");
    if (!fTime)
    {
        *fLog << err << "MTime not found... aborting." << endl;
        return kFALSE;
    }

    fRate = (MEventRate*)pList->FindCreateObj("MEventRate");
    if (!fRate)
        return kFALSE;

    fEvtNumber = 0;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MCerPhotEvt container.
//
#include <TSystem.h>
Int_t MEventRateCalc::Process()
{
    fTime->SetTime(gSystem->Now());

    const ULong_t exec = GetNumExecutions();

    if (fEvtNumber>0)
    {
        if (exec<fEvtNumber)
            return kTRUE;

        const Double_t rate = (Double_t)fNumEvents/(*fTime - fEvtTime);

        *fLog << inf << "Event Rate [Hz]: " << rate << endl;

        fRate->SetRate(rate);
        fRate->SetReadyToSave();
    }

    fEvtNumber = exec+fNumEvents;
    fEvtTime   = *fTime;

    return kTRUE;
}
