/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek, 11/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MFRandomSplit
//
//  A filter which gives fResult = kTRUE with the probability fProb
//
/////////////////////////////////////////////////////////////////////////////
#include "MFRandomSplit.h"

#include <fstream>
#include <TRandom.h>

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MFRandomSplit);

using namespace std;

// --------------------------------------------------------------------------
//
//     Constructor
//
MFRandomSplit::MFRandomSplit(Double_t f, const char *name, const char *title)
    : fProb(f)
{
    fName  = name  ? name  : "MFRandomSplit";
    fTitle = title ? title : "Filter for random splitting";

    if (fProb < 0)
    {
        *fLog << warn << "WARNING - MFRandomSplit::MFRandomSplit: Probability less than 0... set to 0." << endl;
        fProb = 0;
    }

    if (fProb > 1)
    {
        *fLog << warn << "WARNING - MFRandomSplit::MFRandomSplit: Probability greater than 1... set to 1." << endl;
        fProb = 1;
    }
}

// --------------------------------------------------------------------------
//
//  PreProcess. Set fNumSelectedEvts=0
//
Int_t MFRandomSplit::PreProcess(MParList *pList)
{
    fNumSelectedEvts = 0;
    return kTRUE;
}

// --------------------------------------------------------------------------
//  
//  Select events randomly according to the probability fProb. Count all
//  selected events
//
Int_t MFRandomSplit::Process()
{
    fResult = gRandom->Uniform() < fProb;

    if (fResult)
        fNumSelectedEvts++;

    return kTRUE;
}


// --------------------------------------------------------------------------
//  
//  PostProcess. Prints execution statistics
//
Int_t MFRandomSplit::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fNumSelectedEvts << " (";
    *fLog << setw(3) << (int)(100.*fNumSelectedEvts/GetNumExecutions());
    *fLog << "%) selected - out of " << GetNumExecutions() << endl;
    *fLog << endl;

    return kTRUE;
}
