//
// This File contains the definition of the MGCoordinates-class
//
//   Author: Thomas Bretz
//   Version: V1.0 (1-8-2000)

#include "MGSkyPosition.h"

#include <iostream.h>  // cout

#include <TArc.h>
#include <TLine.h>
#include <TText.h>
#include <TList.h>
#include <TCanvas.h>

#include "coord.h"
#include "MAstro.h"
#include "SlaStars.h"
#include "SlaPlanets.h"

ClassImp(MGSkyPosition);

void MGSkyPosition::InitArc(TArc *arc, Int_t fillstyle, Int_t fillcolor, Int_t linecolor)
{
    arc->SetFillStyle(fillstyle);      // (s. TAttFill)
    arc->SetFillColor(fillcolor);      // (s. TAttFill)
    arc->SetLineColor(linecolor);
    arc->Draw();
}

void MGSkyPosition::InitPlanets()
{
    //
    // Magnitudes:
    // -----------
    //   Moon     -12.8
    //   Jupiter   -4.6
    //   Venus     -4.0
    //   Mars      -2.0
    //   Merkur     0.0
    //   Saturn     0.7
    //   Uranus     5.5
    //   Neptun     7.8
    //
    fSlaPlanet = new SlaPlanets(fObservatory);
    fSlaStar   = new SlaStars(fObservatory);

    //
    // Colors: black .. white:
    //
    //   1, 12, 13, 14, 15, 16, 17, 18, 19, 10
    //
    fMars    = new TArc(0, 0, 1);
    fVenus   = new TArc(0, 0, 2);
    fJupiter = new TArc(0, 0, 2);
    fMoon    = new TArc(0, 0, 3, 290, 70);
    fSun     = new TArc(0, 0, 2);
    fSaturn  = new TArc(0, 0, 1);

    fSaturnRing = new TArc(0, 0, 2);
    fSaturnRing->SetFillStyle(4000);      // (s. TAttFill)
    fSaturnRing->SetLineColor(kRed);
    fSaturnRing->Draw();

    InitArc(fMars,    1001, kRed/*13*/, kRed/*12*/);
    InitArc(fVenus,   1001, 15, 14);
    InitArc(fJupiter, 1001, 17, 16);
    InitArc(fMoon,    1001, 19, 15);
    InitArc(fSaturn,  1001, kYellow/*17*/, 16);

    for (int i=0; i<14; i++)
    {
        fStars[i] = new TArc(0, 0, 1);
        InitArc(fStars[i], 1001, 17, 15);
        fList->Add(fStars[i]);
    }

    for (int i=0; i<4; i++)
    {
        fSunL[i] = new TLine(0,0,0,0);
        fSunL[i]->SetLineColor(10);    // white gray
        fSunL[i]->SetLineStyle(1);     // solid  (s. TAttLine)
        fSunL[i]->Draw();

        fList->Add(fSunL[i]);
    }
    InitArc(fSun, 1001, kYellow/*10*/, kYellow/*19*/);

    fList->Add(fMars);
    fList->Add(fVenus);
    fList->Add(fJupiter);
    fList->Add(fMoon);
    fList->Add(fSun);
    fList->Add(fSaturn);

    //
    // Reset 'static' positions of planets
    //
    /*
    for (int i=0; i<9; i++)
    {
        fPlanetX[i] = ~0;
        fPlanetY[i] = ~0;
    }
    for (int i=0; i<13; i++)
    {
        fStarX[i] = ~0;
        fStarY[i] = ~0;
    }*/
}

static const RaDec stars[14] =
{
    // 32349   Sirius           -1.44 1.45    2.64      0.009   A0m
    (RaDec(MAstro::Hms2Rad(  6, 45,  0), MAstro::Dms2Rad(-16, 42,  0))),
    // 30438   Canopus          -0.62 -5.53   95.88     0.164   F0Ib
    (RaDec(MAstro::Hms2Rad(  6, 23,  0), MAstro::Dms2Rad(-52, 41,  0))),
    // 69673   Arcturus         -0.05 -0.31   11.25     1.239   K2IIIp
    (RaDec(MAstro::Hms2Rad( 14, 15,  0), MAstro::Dms2Rad( 19, 10,  0))),
    //         Vega              0.03
    (RaDec(MAstro::Hms2Rad( 18, 37,  0), MAstro::Dms2Rad( 38, 47,  0))),
    //         Altair            0.77
    (RaDec(MAstro::Hms2Rad( 19, 50,  0), MAstro::Dms2Rad(  8, 52,  6))),
    //         Antares           0.96
    (RaDec(MAstro::Hms2Rad( 16, 29, 25), MAstro::Dms2Rad(-26, 25, 55))),
    //         Spica             0.98
    (RaDec(MAstro::Hms2Rad( 13, 25, 13), MAstro::Dms2Rad(-11,  9, 41))),
    //         Deneb             1.25
    (RaDec(MAstro::Hms2Rad( 20, 41, 26), MAstro::Dms2Rad( 45, 16, 49))),
    (RaDec(MAstro::Hms2Rad(  1, 37,  6), MAstro::Dms2Rad(-57, 14,  0))),
    (RaDec(MAstro::Hms2Rad( 12, 26,  6), MAstro::Dms2Rad(-63, 06,  0))),
    (RaDec(MAstro::Hms2Rad(  5, 14,  0), MAstro::Dms2Rad(- 8, 12,  5))),
    (RaDec(MAstro::Hms2Rad(  5, 16,  0), MAstro::Dms2Rad( 46, 00,  0))),
    (RaDec(MAstro::Hms2Rad(  7, 39,  0), MAstro::Dms2Rad(  5, 14,  0))),
    (RaDec(MAstro::Hms2Rad(  5, 55,  0), MAstro::Dms2Rad(  7, 24,  0)))
};

void MGSkyPosition::UpdateStars()
{
    for (int i=0; i<14; i++)
        SetDot(fStars[i], stars[i], 0);
}

void MGSkyPosition::InitText()
{
    fText1 = new TText(-105, 105, "");
    fText2 = new TText(105, 105, "");

    fText1->SetTextAlign(13);  // left, top
    fText2->SetTextAlign(33);  // right, top

    fText1->SetTextColor(10);  // white
    fText2->SetTextColor(10);  // white

    fText1->Draw();
    fText2->Draw();

    fList->Add(fText1);
    fList->Add(fText2);
}

void MGSkyPosition::DrawCoordinateSystem()
{
    TArc arc;
    arc.SetFillStyle(4000);  // transparent
    arc.SetLineColor(13);    // dark gray
    arc.SetLineStyle(1);     // solid  (s. TAttLine)
    arc.DrawArc(0, 0, 90);
    arc.DrawArc(0, 0, 67.5);
    arc.DrawArc(0, 0, 45);
    arc.DrawArc(0, 0, 22.5);

    //
    // FIXME? Use TAxis?
    //
    TLine line;
    line.SetLineColor(1);  // black
    line.SetLineStyle(1);  // solid  (s. TAttLine)
    line.DrawLine(-95,   0, 95,  0);
    line.DrawLine(  0, -95,  0, 95);

    TText text;
    text.SetTextAlign(22);  // centered, centered (s.TAttText)
    text.SetTextColor(13);  // dark gray
    text.DrawText(0.71*22.5, 0.71*22.5, "22.5\xb0");
    text.DrawText(0.71*45.0, 0.71*45.0, "45\xb0");
    text.DrawText(0.71*67.5, 0.71*67.5, "67.5\xb0");
    text.DrawText(0.71*90.0, 0.71*90.0, "90\xb0");

    text.SetTextAlign(22);  // centered, centered (s.TAttText)
    text.SetTextColor(1);   // black

    text.DrawText(0,  102, "N (0\xb0)");
    text.DrawText(0, -102, "S (180\xb0)");

    text.SetTextAngle(270);
    text.DrawText( 102, 0, "E (90\xb0)");

    text.SetTextAngle(90);
    text.DrawText(-102, 0, "W (270\xb0)");
}

void MGSkyPosition::InitPosition()
{
    fLin1a = new TLine(0, 0, 0, 0);
    fLin1b = new TLine(0, 0, 0, 0);
    fLin2  = new TLine(0, 0, 0, 0);

    fLin1a->SetLineColor(5);  // yellow (s. TAttFill)
    fLin1b->SetLineColor(5);  // yellow
    fLin2 ->SetLineColor(10); // white
    fLin1a->SetLineStyle(1);  // solid  (s. TAttLine)
    fLin1b->SetLineStyle(2);  // dashed

    fLin1a->Draw();
    fLin1b->Draw();
    fLin2->Draw();

    fList->Add(fLin1a);
    fList->Add(fLin1b);
    fList->Add(fLin2);

    for (int i=0; i<6; i++)
    {
        fDot[i] = new TArc(0, 0, 1);
        InitArc(fDot[i], 1001, 10, 19);
        fList->Add(fDot[i]);
    }
}

MGSkyPosition::MGSkyPosition(MObservatory::LocationName_t key, const TGWindow* p, const UInt_t w)
    : MGEmbeddedCanvas("SkyPosition", p, w, 110), fObservatory(key)
{
    DrawCoordinateSystem();
    InitPlanets();
    InitPosition();
    InitText();
    InitCanvas();
    SetNoContextMenu();
}

MGSkyPosition::~MGSkyPosition()
{
    delete fSlaPlanet;
    delete fSlaStar;

    cout << "MGSkyPosition destroyed." << endl;
}

void MGSkyPosition::SetLin1(Float_t x, Float_t y)
{
    fLin1a->SetX2(x);
    fLin1a->SetY2(y);
    fLin1b->SetX2(-x);
    fLin1b->SetY2(-y);
}

void MGSkyPosition::SetLin2(Float_t x1, Float_t y1, Float_t x2, Float_t y2)
{
    fLin2->SetX1(x1);
    fLin2->SetX2(x2);
    fLin2->SetY1(y1);
    fLin2->SetY2(y2);
}

void MGSkyPosition::SetDot(TArc *arc, const RaDec &radec, Int_t off)
{
    RaDec rd(radec.Ra()+off*360/24*kDeg2Rad, radec.Dec());
    ZdAz zdaz = fSlaStar->CalcZdAzFast(rd);

    const float s = sin(zdaz.Az());
    const float c = cos(zdaz.Az());

    float x = s*zdaz.Zd()*kRad2Deg;             // [deg]
    float y = c*zdaz.Zd()*kRad2Deg;             // [deg]

    SetDotRange(arc, x, y);
}

void MGSkyPosition::UpdatePosition(RaDec &radec, Float_t zd, Float_t az)
{
    //
    // zd [deg]
    // az [deg]
    //
    static int X =~0;
    static int Y =~0;
    static int Rx=~0;
    static int Ry=~0;

    const float rad = az*kDeg2Rad;    // [rad]

    const float s = sin(rad);
    const float c = cos(rad);

    const float x = s*zd;             // [deg]
    const float y = c*zd;             // [deg]

    const int pixx = (int)(x/fPix);
    const int pixy = (int)(y/fPix);

    const int rx = (int)(s*fWidth/2.);
    const int ry = (int)(c*fWidth/2.);

    if (X==pixx && Y==pixy && Rx==rx && Ry==ry)
        return;

    X = pixx;
    Y = pixy;

    Rx = rx;
    Ry = ry;

    const float dx = s*4.;
    const float dy = c*4.;

    SetLin1(s*95., c*95.);
    SetLin2(x+dy, y-dx, x-dy, y+dx);

    for(int i=0; i<3; i++)
        SetDot(fDot[i], radec, i-3);
    for(int i=3; i<6; i++)
        SetDot(fDot[i], radec, i-2);

    SetModified();
}

void MGSkyPosition::UpdateText(Float_t x, Float_t y)
{
    static int X = ~0;
    static int Y = ~0;

    UShort_t xd, yd;
    Char_t   sx, sy;
    Double_t xm, ym;
    MAstro::Deg2Dm(x, sx, xd, xm);
    MAstro::Deg2Dm(y, sy, yd, ym);

    const int fx = (int)(x*600);
    const int fy = (int)(y*600);

    if (X==fx && Y==fy)
        return;

    X = fx;
    Y = fy;

    char txt[100];
    sprintf(txt, "Zd=%s%d\xb0%02.1f'", sx=='-'?"-":"", xd, xm);
    fText1->SetText(fText1->GetX(), fText1->GetY(), txt);

    sprintf(txt, "Az=%s%d\xb0%02.1f'", sy=='-'?"-":"", yd, ym);
    fText2->SetText(fText2->GetX(), fText2->GetY(), txt);

    SetModified();
}


void MGSkyPosition::Update(ZdAz &pos, double mjd)
{
    //
    // calculate actual time for planet positions
    //
    fSlaPlanet->SetMjd(mjd);
    fSlaStar->SetMjd(mjd);

    UpdatePlanet(kESun,     fSun);
    UpdatePlanet(kEMoon,    fMoon);
    UpdatePlanet(kEJupiter, fJupiter);
    UpdatePlanet(kEVenus,   fVenus);
    UpdatePlanet(kEMars,    fMars);
    UpdatePlanet(kESaturn,  fSaturn);

    UpdateStars();

    RaDec radec = fSlaStar->CalcRaDecFast(pos*kDeg2Rad);
    UpdatePosition(radec, pos.Zd(), pos.Az());
    UpdateText(pos.Zd(), pos.Az());

    UpdateCanvas();
}

bool MGSkyPosition::SetDotRange(TArc *arc, float &x, float &y)
{
    if (!(x>-95 && x<95 && y>-95 && y<95))
    {
        if (arc->GetX1()==-100 && arc->GetY1()==100)
            return false;

        x= 100;
        y=-100;
    }

    const int pixx = (int)(x/fPix);
    const int pixy = (int)(y/fPix);

    const int oldx = (int)(arc->GetX1()/fPix);
    const int oldy = (int)(arc->GetY1()/fPix);

    if (pixx==oldx && pixy==oldy)
        return false;

    arc->SetX1(x);
    arc->SetY1(y);

    SetModified(); 

    return true;
}

void MGSkyPosition::UpdatePlanet(Int_t planet, TArc *arc)
{
    //
    // calculate actual position of planet
    //
    fSlaPlanet->UpdatePlanetPos((ePlanets_t)planet);

    const ZdAz zdaz = fSlaPlanet->GetPlanetPos((ePlanets_t)planet);

    //
    // check whether we have to change the position where it is drawn
    //
    const float s = sin(zdaz.Az());
    const float c = cos(zdaz.Az());

    const float deg = zdaz.Zd()*kRad2Deg;

    float x = s*deg;
    float y = c*deg;

    if (!SetDotRange(arc, x, y))
        return;

    if (planet==kESaturn)
    {
        fSaturnRing->SetX1(x);
        fSaturnRing->SetY1(y);
    }

    if (planet==kESun)
    {
        fSunL[0]->SetX1(x-3.5); fSunL[0]->SetX2(x+3.5);
        fSunL[1]->SetX1(x-2.5); fSunL[1]->SetX2(x+2.5);
        fSunL[2]->SetX1(x-2.5); fSunL[2]->SetX2(x+2.5);
        fSunL[3]->SetX1(x);     fSunL[3]->SetX2(x);

        fSunL[0]->SetY1(y);     fSunL[0]->SetY2(y);
        fSunL[1]->SetY1(y-2.5); fSunL[1]->SetY2(y+2.5);
        fSunL[2]->SetY1(y+2.5); fSunL[2]->SetY2(y-2.5);
        fSunL[3]->SetY1(y-3.5); fSunL[3]->SetY2(y+3.5);
    }
}
