/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug  09/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//   MExtractSignal                                                         //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MExtractSignal.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include "MGeomCam.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtData.h" 
#include "MRawEvtPixelIter.h"

#include "TMath.h"

ClassImp(MExtractSignal);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
MExtractSignal::MExtractSignal(const Byte_t first, const Byte_t last, const Byte_t logainshift, 
                    const char *name, const char *title)
  : fNumHiGainSamples(last-first+1), fNumLoGainSamples(last-first+1),
    fSaturationLimit(254), fConversionHiLo(10.)
{

    fName  = name  ? name  : "MExtractSignal";
    fTitle = title ? title : "Task to extract the signal from the FADC slices";

    AddToBranchList("MRawEvtData.*");

    fFirst       = first;
    fLoGainFirst = first+logainshift;

    fSqrtHiGainSamples = TMath::Sqrt((Float_t)fNumHiGainSamples); 
    fSqrtLoGainSamples = TMath::Sqrt((Float_t)fNumLoGainSamples); 
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MExtractedSignalCam
//
Int_t MExtractSignal::PreProcess(MParList *pList)
{

    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }


    fSignals = (MExtractedSignalCam*)pList->FindCreateObj("MExtractedSignalCam");
    if (!fSignals)
        return kFALSE;

    fPedestals = (MPedestalCam*)pList->FindObject("MPedestalCam");
    if (!fPedestals)
      {
        *fLog << err << dbginf << "Cannot find MPedestalCam ... aborting" << endl;
        return kFALSE;
      }
    
    return kTRUE;
}


// --------------------------------------------------------------------------
//
// The ReInit searches for the following input containers:
//  - MRawRunHeader
//
Bool_t MExtractSignal::ReInit(MParList *pList )
{
 

    MGeomCam *cam = (MGeomCam*)pList->FindObject(AddSerialNumber("MGeomCam"));
    if (!cam)
    {
        *fLog << err << GetDescriptor() << ": No MGeomCam found... aborting." << endl;
        return kFALSE;
    }

    fSignals->InitSize(cam->GetNumPixels());    


    fSignals->SetNumUsedFADCSlices(fNumHiGainSamples,fFirst,fFirst+fNumHiGainSamples-1,
				   fNumLoGainSamples,fFirst,fFirst+fNumLoGainSamples-1);

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MExtractedSignalCam container.
//
Int_t MExtractSignal::Process()
{

    MRawEvtPixelIter pixel(fRawEvt);
    fSignals->Clear();

    while (pixel.Next())
      {

        UShort_t satHi = 0;
        UShort_t satLo = 0;

        Byte_t *ptr   = pixel.GetHiGainSamples();
        Byte_t *first = ptr + fFirst;
        Byte_t *last  = ptr + fFirst + fNumHiGainSamples;

	UInt_t  sumHi   = 0;
	UInt_t  sumLo   = 0;

        const Int_t pixid = pixel.GetPixelId();
        const MPedestalPix  &ped = (*fPedestals)[pixid]; 
	MExtractedSignalPix &pix = (*fSignals)[pixid];

        const Float_t pedes   = ped.GetPedestal();
        const Float_t pedrms  = ped.GetPedestalRms();

	Byte_t maxhi  = 0;
	Byte_t midhi  = 0;

	for (ptr=first;ptr<last;ptr++)
	  {
	    
	    if (*ptr > maxhi)
	      {
		maxhi = *ptr;
		midhi = ptr-first;
	      }

	    sumHi += *ptr;
	    
	    if (*ptr >= fSaturationLimit)
	      satHi++;

	  }

	ptr   = pixel.GetLoGainSamples();
        first = ptr + fLoGainFirst + 1;
        last  = ptr + fLoGainFirst + fNumLoGainSamples +1;
	Byte_t maxlo  = 0;
	Byte_t midlo  = 0;
	
	for (ptr=first;ptr<last;ptr++)
	  {

	    if (*ptr > maxlo)
	      {
		maxlo = *ptr;
		midlo = ptr-first;
	      }

	    sumLo += *ptr;

	    if (*ptr >= fSaturationLimit)
	      {
		*fLog << err << dbginf 
		      << "Warning: Saturation of Lo Gain reached in pixel: " 
		      << pixid << " " << "   sum = " << sumLo << endl;
		satLo++;
	      }
	  }

	pix.SetExtractedSignal((Float_t)sumHi - pedes*(Float_t)fNumHiGainSamples,
			       pedrms*fSqrtHiGainSamples,
			       ((Float_t)sumLo - pedes*(Float_t)fNumLoGainSamples)*fConversionHiLo,
			       pedrms*fSqrtLoGainSamples
			       );

	pix.SetGainSaturation((satHi),satHi,satLo);

	if (satHi)
	  pix.SetArrivalTimes((Float_t)(midlo+fFirst),0.,0.);
	else
	  pix.SetArrivalTimes((Float_t)(midhi+fLoGainFirst),0.,0.);

      } /* while (pixel.Next()) */


    fSignals->SetReadyToSave();

    return kTRUE;
}

Int_t MExtractSignal::PostProcess()
{

  return kTRUE;

}
