/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 12/2000 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MMcPedestalCopy
//
//  This task looks for the nformation about FADC pedestals in
//  MMcFadcHeader and translates it to the pedestal values in
//  MPedestalCam
//
//  Input Containers:
//   MMcFadcHeader
//   [MMcRunHeader]
//   [MRawRunHeader]
//
//  Output Containers:
//   MPedestalCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MMcPedestalCopy.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalPix.h"
#include "MPedestalCam.h"

#include "MRawRunHeader.h"
#include "MMcRunHeader.hxx"
#include "MMcFadcHeader.hxx"

ClassImp(MMcPedestalCopy);

using namespace std;

MMcPedestalCopy::MMcPedestalCopy(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalCopy";
    fTitle = title ? title : "Copy MC pedestals into MPedestal Container";

    //
    // This is not needed here using MReadMarsFile because for the
    // RunHeader tree the auto scheme is disabled by default
    //
    AddToBranchList("MMcFadcHeader.fPedesMean");
    AddToBranchList("MMcFadcHeader.fElecNoise");
}

// --------------------------------------------------------------------------
//
// Check for the run type. Return kTRUE if it is a MC run or if there
// is no MC run header (old camera files) kFALSE in case of a different
// run type
//
Bool_t MMcPedestalCopy::CheckRunType(MParList *pList) const
{
    const MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return run->GetRunType() == kRTMonteCarlo;
}

// --------------------------------------------------------------------------
//
// Make sure, that there is an MPedestalCam Object in the Parameter List.
//
Int_t MMcPedestalCopy::PreProcess(MParList *pList)
{
  //
  // If it is no MC file skip this function...
  //
  if (! CheckRunType(pList))
    return kTRUE;

  return pList->FindCreateObj(AddSerialNumber("MPedestalCam")) ? kTRUE : kFALSE;
}

// --------------------------------------------------------------------------
//
// Check for the runtype.
// Search for MPedestalCam and MMcFadcHeader.
//
Bool_t MMcPedestalCopy::ReInit(MParList *pList)
{
    //
    // If it is no MC file skip this function...
    //
    if (!CheckRunType(pList))
        return kTRUE;

    //
    // Now check the existance of all necessary containers. This has
    // to be done only if this is a MC file.
    //
    MMcFadcHeader *fadc = (MMcFadcHeader*)pList->FindObject(AddSerialNumber("MMcFadcHeader"));
    if (!fadc)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    MPedestalCam *pedcam = (MPedestalCam*)pList->FindCreateObj(AddSerialNumber("MPedestalCam"));
    if (!pedcam)
        return kFALSE;

    MMcRunHeader *mcrun = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrun)
        *fLog << warn << dbginf << "MMcRunHeader not found... assuming camera<0.7" << endl;

    const int num = pedcam->GetSize();

    const Bool_t camver70 = mcrun && mcrun->GetCamVersion()>=70;

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix = (*pedcam)[i];

        // Here one should compute the Pedestal taking into account how
        // the MC makes the transformation analogic-digital for the FADC.
        // This is done only once per file -> not time critical.

        const Float_t pedest = fadc->GetPedestal(i);
        const Float_t sigma  = camver70 ? fadc->GetPedestalRmsHigh(i) : fadc->GetElecNoise(i);

        pix.Set(pedest, sigma);

    }

    if (camver70)
        pedcam->SetReadyToSave();

    return kTRUE;
}
