/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalCam                                                            //
//                                                                         //
// Hold the Pedestal information for all pixels in the camera              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

ClassImp(MPedestalCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. Creates a MPedestalPix object for each pixel
//
MPedestalCam::MPedestalCam(const char *name, const char *title)
{
    fName  = name  ? name  : "MPedestalCam";
    fTitle = title ? title : "Storage container for all Pedestal Information in the camera";

    fArray = new TClonesArray("MPedestalPix", 1);

}

// --------------------------------------------------------------------------
//
// Delete the array conatining the pixel pedest information
//
MPedestalCam::~MPedestalCam()
{
    delete fArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MPedestalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// This function returns the current size of the TClonesArray 
// independently if the MPedestalPix is filled with values or not.
//
// Get the size of the MPedestalCam
//
Int_t MPedestalCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i)
{
  return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i) const
{
  return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

void MPedestalCam::Clear(Option_t *o)
{
    fArray->ForEach(TObject, Clear)();
}

void MPedestalCam::InitUseHists()
{
    fArray->ForEach(MPedestalPix, InitUseHists)();
}

void MPedestalCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MPedestalPix *pix;
    while ((pix=(MPedestalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": ";
        *fLog << pix->GetPedestal() << " " << pix->GetPedestalRms() << endl;
    }
}

Float_t MPedestalCam::GetPedestalMin(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

Float_t MPedestalCam::GetPedestalMax(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

Bool_t MPedestalCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  switch (type)
    {
    case 0:
      if ((*this)[idx].IsValid())
        val = (*this)[idx].GetPedestal();
      else 
        return kFALSE;
      break;
    case 1:
      if ((*this)[idx].IsValid())
        val = (*this)[idx].GetPedestalRms();
      else
        return kFALSE;
      break;
    case 2:
      //      if ((*this)[idx].IsFitValid())
        val = (*this)[idx].GetMean();
        //      else
        //        return kFALSE;
      break;
    case 3:
      //      if ((*this)[idx].IsFitValid())
        val = (*this)[idx].GetMeanErr();
        //      else
        //        return kFALSE;
      break;
    case 4:
      //      if ((*this)[idx].IsFitValid())
        val = (*this)[idx].GetSigma();
        //      else
        //        return kFALSE;
      break;
    case 5:
      //      if ((*this)[idx].IsFitValid())
        val = (*this)[idx].GetSigmaErr();
        //      else
        //        return kFALSE;
      break;
    case 6:
      //      if ((*this)[idx].IsFitValid())
        val = (*this)[idx].GetProb();
        //      else
        //        return kFALSE;
      break;
    case 7:
      //      if ((*this)[idx].IsFitValid())
        val = ((*this)[idx].GetPedestal()-(*this)[idx].GetMean());
        //      else
        //        return kFALSE;
      break;
    case 8:
      //      if ((*this)[idx].IsFitValid())
        val = ((*this)[idx].GetPedestalRms()-(*this)[idx].GetSigma());
        //      else
        //        return kFALSE;
      break;
    default:
      return kFALSE;
    }
  return kTRUE;
}

void MPedestalCam::DrawPixelContent(Int_t idx) const
{
  (*this)[idx].Draw();
}
