#ifndef MARS_MCalibrationPINDiode
#define MARS_MCalibrationPINDiode

#ifndef MARS_MHCalibrationPINDiode
#include "MHCalibrationPINDiode.h"
#endif

class MCalibrationPINDiode : public MParContainer
{
private:

  MHCalibrationPINDiode *fHist; // Pointer to the histograms performing the fits, etc.  

  const Float_t fChargeLimit;       // The limit (in units of PedRMS) for acceptance of the fitted mean charge
  const Float_t fChargeErrLimit;    // The limit (in units of PedRMS) for acceptance of the fitted charge sigma
  const Float_t fChargeRelErrLimit; // The limit (in units of Error of fitted charge) for acceptance of the fitted mean  
  
  Float_t fPed;                   // The mean pedestal (from MPedestalPix)
  Float_t fPedRms;                // The pedestal  RMS (from MPedestalPix)

  Float_t fCharge;                // The mean charge after the fit
  Float_t fErrCharge;             // The error of mean charge after the fit
  Float_t fSigmaCharge;           // The sigma of the mean charge after the fit
  Float_t fErrSigmaCharge;        // The error of the sigma of the mean charge after the fit
  Float_t fRSigmaSquare;          // The reduced squares of sigmas after the fit
  Float_t fChargeProb;            // The probability of the fit function 

  Float_t fTime;                  // The mean arrival time after the fit  
  Float_t fSigmaTime;             // The error of the mean arrival time after the fit
  Float_t fTimeChiSquare;         // The Chi Square of the fit function 
  Float_t fTimeProb;              // The probability of the fit function 
  
  Byte_t  fFlags;                 // Flag for the set Bits

  enum  { kExcluded, kExcludeQualityCheck,
          kFitValid, kFitted  };
  
  Bool_t CheckChargeFitValidity();
  Bool_t CheckTimeFitValidity();
  Bool_t CheckOscillations();  
  
public:

  MCalibrationPINDiode(const char *name=NULL, const char *title=NULL);
  ~MCalibrationPINDiode();
  
  void Clear(Option_t *o="");
  
  // Getter
  MHCalibrationPINDiode *GetHist() const   { return fHist;     }

  // Charges
  Float_t GetCharge()              const { return fCharge;         }
  Float_t GetErrCharge()           const { return fErrCharge;      }
  Float_t GetChargeProb()          const { return fChargeProb;     }    
  Float_t GetSigmaCharge()         const { return fSigmaCharge;    }
  Float_t GetErrSigmaCharge()      const { return fErrSigmaCharge; }
  Float_t GetRSigmaSquare()        const { return fRSigmaSquare;   }

  // Times  
  Float_t GetTime()                const { return fTime;           }
  Float_t GetSigmaTime()           const { return fSigmaTime;      }
  Float_t GetTimeChiSquare()       const { return fTimeChiSquare;  }
  Float_t GetTimeProb()            const { return fTimeProb;  }      

  // Pedestals
  Float_t GetPed()                 const { return fPed;    }
  Float_t GetPedRms()              const { return fPedRms; }

  Bool_t IsExcluded()              const;
  Bool_t IsFitValid()              const;
  Bool_t IsFitted()                const;

  // Setters
  void SetPedestal(Float_t ped, Float_t pedrms);
  void SetExcluded(Bool_t b = kTRUE);
  void SetExcludeQualityCheck(Bool_t b = kTRUE);
  void SetFitValid(Bool_t b = kTRUE);
  void SetFitted(Bool_t b = kTRUE);

  // Fill histos
  Bool_t FillCharge(Float_t q)      { return fHist->FillChargeHiGain(q); }
  Bool_t FillTime(Float_t t)        { return fHist->FillTimeHiGain(t); }  
  Bool_t FillRChargevsTime(Float_t rq, Int_t t) { return fHist->FillChargevsNHiGain(rq,t); }    

  // Fits
  Bool_t FitCharge();
  Bool_t FitTime();

  // Draws
  void Draw(Option_t *opt="")                    { fHist->Draw(opt); }
  TObject *DrawClone(Option_t *opt="") const    { return fHist->DrawClone(opt); }  
  
  ClassDef(MCalibrationPINDiode, 1)	// Storage Container for Calibration information of the PIN Diode
};

#endif   /* MARS_MCalibrationPINDiode */

