/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Sabrina Stark   12/2003 <mailto:lstark@particle.phys.ethz.ch>
!              Wolfgang Wittek 12/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalWorkaround                                                     //
//                                                                         //
// copies the pedestal value and the pedestal RMS                          //
//        from the container MPedPhotCam                                   //
//        into the contaier  MPedestalCam                                  //
//                                                                         //
// put the zenith angle into MMcEvt;                                       //
//     the zenith angle is taken from the runbooks                         //
//     (1 fixed zenith angle for a given run)                              //
//                                                                         //
// this workaround is necessary                                            //
// - as long as the analysis classes                                       //
//   take the pedestals from MPedestalCam; in the long run they have to    //
//   be taken from MPedPhotCam                                             //
// - and as long as the container for the zenith angle is not defined      //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalWorkaround.h"

#include <stdio.h>
#include "MLog.h"
#include "MLogManip.h"
#include "MParList.h"
#include "MGeomCam.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"
#include "MPedPhotCam.h"
#include "MMcEvt.hxx"
#include "MRawRunHeader.h"

using namespace std;

ClassImp(MPedestalWorkaround);

MPedestalWorkaround::MPedestalWorkaround(const char *name, const char *title)
{
    fName = name ? name : "MPedestalWorkaround";
    fTitle = title ? title : "Storage of pedestal values and RMS in units of photons";
   
}

// ------------------------------------------------------------------------
//

Int_t MPedestalWorkaround::PreProcess(MParList *pList)
{
   fPed = (MPedestalCam*)pList->FindObject("MPedestalCam");
   if (!fPed)
     {
       *fLog << err << "MPedestalCam not found... aborting." << endl;
       return kFALSE;
     }

   fPedPhot = (MPedPhotCam*)pList->FindObject("MPedPhotCam");
   if (!fPedPhot)
     {
       *fLog << err << "MPedPhotCam not found... aborting." << endl;
       return kFALSE;
     }

   fCam = (MGeomCam*)pList->FindObject("MGeomCam");
   if (!fCam)
     {
       *fLog << err << "MGeomCam not found (no geometry information available)... aborting." << endl;
       return kFALSE;
     }

   fRun = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
   if (!fRun)
   {
       *fLog << err << "MRawRunHeader not found... aborting." << endl;
       return kFALSE;
   }

   fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
   if (!fMcEvt)
   {
       *fLog << err << "MMcEvt not found... aborting." << endl;
       return kFALSE;
   }

   return kTRUE;
}

// ------------------------------------------------------------------------
//
Int_t MPedestalWorkaround::Process()
{
  //-------------------------------------------------------------------
  // copy the pedestal and the pedestal RMS 
  // from MPedPhotCam into MPedestalCam   

  UInt_t imaxnumpix = fCam->GetNumPixels();

  for (UInt_t i=0; i<imaxnumpix; i++)
  {
    Int_t type = 0;
    Double_t val;
    Float_t valout;
    fPedPhot->GetPixelContent( val, i, *fCam, type);
    valout = (*fPed)[i].GetPedestal();
    (*fPed)[i].SetPedestal(val);
    *fLog << "i, val, valout : " << i <<",  "<<  val<<",  " << valout << endl;
    type = 1;
    fPedPhot->GetPixelContent( val, i, *fCam, type);
    valout = (*fPed)[i].GetPedestalRms();
    (*fPed)[i].SetPedestalRms(val);
    *fLog << "RMS : i, val, valout : " << i <<",  "<<  val<<",  " << valout 
          << endl;
  }


  //-------------------------------------------------------------------
  // put the zenith angle into MMcEvt

  Int_t run = fRun->GetRunNumber();
  Double_t thetadeg;
  Double_t thetarad;

  if      (run == 3127)  thetadeg = 27.2;  // Crab
  else if (run == 3128)  thetadeg = 25.6;
  else if (run == 3129)  thetadeg = 24.3;
  else if (run == 3130)  thetadeg = 23.9;


  else if (run == 3132)  thetadeg = 22.9;
  else if (run == 3133)  thetadeg = 22.8;
  else if (run == 3134)  thetadeg = 22.3;
  else if (run == 3135)  thetadeg = 21.9;
  else if (run == 3136)  thetadeg = 21.5;
  else if (run == 3137)  thetadeg = 21.1;
  else if (run == 3138)  thetadeg = 20.8;
  else if (run == 3139)  thetadeg = 20.4;

  else if (run == 3140)  thetadeg = 19.5;
  else if (run == 3141)  thetadeg = 19.4;
  else if (run == 3142)  thetadeg = 19.0;
  else if (run == 3143)  thetadeg = 18.6;
  else if (run == 3144)  thetadeg = 13.0;
  else if (run == 3145)  thetadeg = 12.4;
  else if (run == 3146)  thetadeg = 12.1;
  else if (run == 3147)  thetadeg = 11.7;
  else if (run == 3148)  thetadeg = 11.3;
  else if (run == 3149)  thetadeg = 11.9;

  else if (run == 3150)  thetadeg = 10.6;
  else if (run == 3151)  thetadeg = 10.3;
  else if (run == 3152)  thetadeg = 10.0;
  else if (run == 3153)  thetadeg =  9.6;
  else if (run == 3154)  thetadeg =  9.3;
  else if (run == 3155)  thetadeg =  9.0;
  else if (run == 3156)  thetadeg =  8.7;
  else if (run == 3157)  thetadeg =  8.4;
  else if (run == 3158)  thetadeg =  8.1;
  else if (run == 3159)  thetadeg =  7.9;

  else if (run == 3160)  thetadeg =  7.7;
  else if (run == 3161)  thetadeg =  7.3;
  else if (run == 3162)  thetadeg =  7.2;
  else if (run == 3163)  thetadeg =  7.0;
  else if (run == 3164)  thetadeg =  6.8;
  else if (run == 3165)  thetadeg =  6.7;
  else if (run == 3166)  thetadeg =  6.6;
  else if (run == 3167)  thetadeg =  6.5;
  else if (run == 3168)  thetadeg =  6.4;
  else if (run == 3169)  thetadeg =  6.4;

  else if (run == 3170)  thetadeg =  6.4;
  else if (run == 3171)  thetadeg =  6.4;
  else if (run == 3172)  thetadeg =  6.5;
  else if (run == 3173)  thetadeg =  6.6;
  else if (run == 3174)  thetadeg =  6.7;

  else if (run == 3176)  thetadeg =  7.1;
  else if (run == 3177)  thetadeg =  7.4;
  else if (run == 3178)  thetadeg =  7.6;
  else if (run == 3179)  thetadeg =  7.9;


  else if (run == 3182)  thetadeg =  8.4;
  else if (run == 3183)  thetadeg =  8.9;
  else if (run == 3184)  thetadeg =  9.2;
  else if (run == 3185)  thetadeg =  9.5;
  else if (run == 3186)  thetadeg =  9.8;
  else if (run == 3187)  thetadeg = 10.5;
  else if (run == 3188)  thetadeg = 10.9;
  else if (run == 3189)  thetadeg = 11.2;

  else if (run == 3190)  thetadeg = 11.6;
  else if (run == 3191)  thetadeg = 11.6;
  else if (run == 3192)  thetadeg = 12.4;
  else if (run == 3193)  thetadeg = 12.7;
  else if (run == 3194)  thetadeg = 13.1;
  else if (run == 3195)  thetadeg = 13.5;
  else if (run == 3196)  thetadeg = 13.9;
  else if (run == 3197)  thetadeg = 14.3;
  else if (run == 3198)  thetadeg = 14.7;
  else if (run == 3199)  thetadeg = 15.1;

  else if (run == 3200)  thetadeg = 15.6;
  else if (run == 3201)  thetadeg = 16.0;
  else if (run == 3202)  thetadeg = 16.5;
  else if (run == 3203)  thetadeg = 16.9;
  else if (run == 3204)  thetadeg = 17.3;
  else if (run == 3205)  thetadeg = 17.7;
  else if (run == 3206)  thetadeg = 18.2;
  else if (run == 3207)  thetadeg = 18.6;
  else if (run == 3208)  thetadeg = 19.0;
  else if (run == 3209)  thetadeg = 19.4;

  else if (run == 3210)  thetadeg = 19.9;
  else if (run == 3211)  thetadeg = 20.4;
  else if (run == 3212)  thetadeg = 20.8;
  else if (run == 3213)  thetadeg = 21.2;
  else if (run == 3214)  thetadeg = 21.7;
  else if (run == 3215)  thetadeg = 22.2;
  else if (run == 3216)  thetadeg = 25.6;  // Off Crab1
  else if (run == 3217)  thetadeg = 25.0;
  else if (run == 3218)  thetadeg = 24.5;
  else if (run == 3219)  thetadeg = 24.0;

  else if (run == 3220)  thetadeg = 23.5;
  else if (run == 3221)  thetadeg = 22.5;
  else if (run == 3222)  thetadeg = 22.1;

  else if (run == 3225)  thetadeg = 15.1;
  else if (run == 3226)  thetadeg = 15.0;
  else if (run == 3227)  thetadeg = 14.5;
  else if (run == 3228)  thetadeg = 14.1;
  else if (run == 3229)  thetadeg = 13.8;

  else if (run == 3230)  thetadeg = 13.3;
  else if (run == 3231)  thetadeg = 13.0;
  else if (run == 3232)  thetadeg = 12.6;
  else if (run == 3233)  thetadeg = 12.3;
  else if (run == 3234)  thetadeg = 12.0;
  else if (run == 3235)  thetadeg = 11.6;
  else if (run == 3236)  thetadeg = 11.3;
  else if (run == 3237)  thetadeg = 11.0;
  else if (run == 3238)  thetadeg = 10.8;
  else if (run == 3239)  thetadeg = 10.4;

  else if (run == 3240)  thetadeg = 10.1;
  else if (run == 3241)  thetadeg =  9.9;
  else if (run == 3242)  thetadeg =  9.6;
  else if (run == 3243)  thetadeg =  9.4;
  else if (run == 3244)  thetadeg =  9.2;
  else if (run == 3245)  thetadeg =  9.0;
  else if (run == 3246)  thetadeg =  8.9;
  else if (run == 3247)  thetadeg =  8.8;
  else if (run == 3248)  thetadeg =  8.7;
  else if (run == 3249)  thetadeg =  8.6;

  else if (run == 3250)  thetadeg =  8.6;
  else if (run == 3251)  thetadeg =  8.6;
  else if (run == 3252)  thetadeg =  8.6;
  else if (run == 3253)  thetadeg =  8.7;
  else if (run == 3254)  thetadeg =  8.8;
  else if (run == 3255)  thetadeg =  8.9;
  else if (run == 3256)  thetadeg =  9.1;
  else if (run == 3257)  thetadeg =  9.3;
  else if (run == 3258)  thetadeg =  9.5;
  else if (run == 3259)  thetadeg =  9.7;
  else if (run == 3260)  thetadeg =  9.9;

  else if (run == 3261)  thetadeg = 10.2;
  else if (run == 3262)  thetadeg = 10.5;
  else if (run == 3263)  thetadeg = 10.8;
  else if (run == 3264)  thetadeg = 11.1;
  else if (run == 3265)  thetadeg = 11.4;
  else if (run == 3266)  thetadeg = 11.8;
  else if (run == 3267)  thetadeg = 12.1;
  else if (run == 3268)  thetadeg = 12.5;
  else if (run == 3269)  thetadeg = 12.8;

  else if (run == 3270)  thetadeg = 13.2;
  else if (run == 3271)  thetadeg = 13.5;
  else if (run == 3272)  thetadeg = 13.9;
  else if (run == 3273)  thetadeg = 14.0;
  else if (run == 3274)  thetadeg = 14.4;

  else if (run == 3284)  thetadeg =  7.0;  // Crab
  else if (run == 3285)  thetadeg =  7.2;
  else if (run == 3286)  thetadeg =  7.4;
  else if (run == 3287)  thetadeg =  7.5;
  else if (run == 3288)  thetadeg =  8.4;
  else if (run == 3289)  thetadeg =  9.0;

  else if (run == 3290)  thetadeg =  9.4;
  else if (run == 3291)  thetadeg =  9.8;
  else if (run == 3292)  thetadeg = 10.2;
  else if (run == 3293)  thetadeg = 10.5;
  else if (run == 3294)  thetadeg = 10.9;
  else if (run == 3295)  thetadeg = 11.3;
  else if (run == 3296)  thetadeg = 11.8;
  else if (run == 3297)  thetadeg = 12.2;
  else if (run == 3298)  thetadeg = 12.6;
  else if (run == 3299)  thetadeg = 13.0;

  else if (run == 3300)  thetadeg = 13.5;
  else if (run == 3301)  thetadeg = 13.9;
  else if (run == 3302)  thetadeg = 14.3;
  else if (run == 3303)  thetadeg = 14.8;
  else if (run == 3304)  thetadeg = 15.2;
  else if (run == 3305)  thetadeg = 15.7;
  else if (run == 3306)  thetadeg = 16.2;
  else if (run == 3307)  thetadeg = 16.6;
  else if (run == 3308)  thetadeg = 17.1;
  else if (run == 3309)  thetadeg = 17.6;

  else if (run == 3310)  thetadeg = 17.9;
  else if (run == 3311)  thetadeg = 18.4;
  else if (run == 3312)  thetadeg = 18.9;
  else if (run == 3313)  thetadeg = 19.3;
  else if (run == 3314)  thetadeg = 19.8;
  else if (run == 3315)  thetadeg = 20.1;
  else if (run == 3316)  thetadeg = 20.7;
  else if (run == 3317)  thetadeg = 21.2;
  else if (run == 3318)  thetadeg = 21.7;
  else if (run == 3319)  thetadeg = 22.1;

  else if (run == 3320)  thetadeg = 22.5;
  else if (run == 3321)  thetadeg = 23.1;
  else if (run == 3322)  thetadeg = 23.6;
  else if (run == 3323)  thetadeg = 24.1;
  else if (run == 3324)  thetadeg = 24.6;
  else if (run == 3325)  thetadeg = 24.9;
  else if (run == 3326)  thetadeg = 25.5;
  else if (run == 3327)  thetadeg = 26.0;
  else if (run == 3328)  thetadeg = 26.0;
  else if (run == 3329)  thetadeg = 26.6;

  else if (run == 3330)  thetadeg = 26.6;
  else if (run == 3331)  thetadeg = 27.1;
  else if (run == 3332)  thetadeg = 27.7;
  else if (run == 3333)  thetadeg = 28.2;
  else if (run == 3334)  thetadeg = 28.5;

  else if (run == 3340)  thetadeg = 10.5;
  else if (run == 3341)  thetadeg = 10.3;
  else if (run == 3342)  thetadeg =  9.6;
  else if (run == 3343)  thetadeg =  9.2;
  else if (run == 3344)  thetadeg =  8.9;
  else if (run == 3345)  thetadeg =  8.6;
  else if (run == 3346)  thetadeg =  8.3;
  else if (run == 3347)  thetadeg =  8.0;
  else if (run == 3348)  thetadeg =  7.7;
  else if (run == 3349)  thetadeg =  7.5;

  else if (run == 3350)  thetadeg =  7.2;
  else if (run == 3351)  thetadeg =  7.0;
  else if (run == 3352)  thetadeg =  6.8;
  else if (run == 3353)  thetadeg =  6.7;
  else if (run == 3354)  thetadeg =  6.6;
  else if (run == 3355)  thetadeg =  6.5;
  else if (run == 3356)  thetadeg =  6.4;
  else if (run == 3357)  thetadeg =  6.4;
  else if (run == 3358)  thetadeg =  6.4;
  else if (run == 3359)  thetadeg =  6.5;

  else if (run == 3360)  thetadeg =  6.6;

  else if (run == 3362)  thetadeg =  6.7;
  else if (run == 3363)  thetadeg =  6.8;
  else if (run == 3364)  thetadeg =  7.0;
  else if (run == 3365)  thetadeg =  7.2;
  else if (run == 3366)  thetadeg =  7.5;
  else if (run == 3367)  thetadeg =  7.7;
  else if (run == 3368)  thetadeg =  8.0;
  else if (run == 3369)  thetadeg =  8.3;

  else if (run == 3370)  thetadeg =  8.6;
  else if (run == 3371)  thetadeg =  9.0;
  else if (run == 3372)  thetadeg =  9.3;
  else if (run == 3373)  thetadeg =  9.6;
  else if (run == 3374)  thetadeg = 10.0;
  else if (run == 3375)  thetadeg = 10.4;
  else if (run == 3376)  thetadeg = 10.7;
  else if (run == 3377)  thetadeg = 11.1;
  else if (run == 3378)  thetadeg = 11.5;
  else if (run == 3379)  thetadeg = 11.9;

  else if (run == 3380)  thetadeg = 12.3;
  else if (run == 3381)  thetadeg = 12.7;
  else if (run == 3382)  thetadeg = 13.1;
  else if (run == 3383)  thetadeg = 13.5;
  else if (run == 3384)  thetadeg = 13.9;
  else if (run == 3385)  thetadeg = 14.3;
  else if (run == 3386)  thetadeg = 14.7;
  else if (run == 3387)  thetadeg = 15.2;
  else if (run == 3388)  thetadeg = 15.6;
  else if (run == 3389)  thetadeg = 16.0;

  else if (run == 3390)  thetadeg = 16.4;
  else if (run == 3391)  thetadeg = 16.7;
  else if (run == 3392)  thetadeg = 17.9;
  else if (run == 3393)  thetadeg = 18.3;
  else if (run == 3394)  thetadeg = 18.7;
  else if (run == 3395)  thetadeg = 19.2;
  else if (run == 3396)  thetadeg = 19.6;
  else if (run == 3397)  thetadeg = 20.0;
  else if (run == 3398)  thetadeg = 20.5;
  else if (run == 3399)  thetadeg = 20.9;

  else if (run == 3400)  thetadeg = 21.4;
  else if (run == 3401)  thetadeg = 21.8;
  else if (run == 3402)  thetadeg = 22.1;
  else if (run == 3403)  thetadeg = 22.6;
  else if (run == 3404)  thetadeg = 23.1;
  else if (run == 3405)  thetadeg = 23.4;
  else if (run == 3406)  thetadeg = 23.9;
  else if (run == 3407)  thetadeg = 24.3;
  else if (run == 3408)  thetadeg = 24.7;
  else if (run == 3409)  thetadeg = 26.9;

  else if (run == 3410)  thetadeg = 27.3;
  else if (run == 3411)  thetadeg = 27.7;
  else if (run == 3412)  thetadeg = 28.2;
  else if (run == 3413)  thetadeg = 28.7;
  else if (run == 3414)  thetadeg = 29.1;
  else if (run == 3415)  thetadeg = 29.2;
  else if (run == 3416)  thetadeg = 30.0;
  else if (run == 3417)  thetadeg = 18.5;  // Off Crab1
  else if (run == 3418)  thetadeg = 18.4;
  else if (run == 3419)  thetadeg = 17.5;

  else if (run == 3420)  thetadeg = 17.2;
  else if (run == 3421)  thetadeg = 16.8;
  else if (run == 3422)  thetadeg = 16.4;
  else if (run == 3423)  thetadeg = 16.0;
  else if (run == 3424)  thetadeg = 15.6;
  else if (run == 3425)  thetadeg = 15.3;
  else if (run == 3426)  thetadeg = 14.9;
  else if (run == 3427)  thetadeg = 14.5;
  else if (run == 3428)  thetadeg = 14.1;
  else if (run == 3429)  thetadeg = 13.7;

  else if (run == 3430)  thetadeg = 13.4;
  else if (run == 3431)  thetadeg = 13.0;
  else if (run == 3432)  thetadeg = 12.7;
  else if (run == 3433)  thetadeg = 12.3;
  else if (run == 3434)  thetadeg = 12.0;
  else if (run == 3435)  thetadeg = 12.0;
  else if (run == 3436)  thetadeg = 11.6;
  else if (run == 3437)  thetadeg = 11.3;
  else if (run == 3438)  thetadeg = 11.0;
  else if (run == 3439)  thetadeg = 10.7;

  else if (run == 3440)  thetadeg = 10.4;
  else if (run == 3441)  thetadeg = 10.1;
  else if (run == 3442)  thetadeg =  9.9;
  else if (run == 3443)  thetadeg =  9.8;
  else if (run == 3444)  thetadeg = 30.8;  // Mkn 421
  else if (run == 3445)  thetadeg = 30.6;
  else if (run == 3446)  thetadeg = 29.7;
  else if (run == 3447)  thetadeg = 29.3;
  else if (run == 3448)  thetadeg = 28.9;
  else if (run == 3449)  thetadeg = 28.5;

  else if (run == 3450)  thetadeg = 28.1;
  else if (run == 3451)  thetadeg = 27.7;
  else if (run == 3452)  thetadeg = 27.3;
  else if (run == 3453)  thetadeg = 26.9;
  else if (run == 3454)  thetadeg = 26.5;
  else if (run == 3455)  thetadeg = 26.1;
  else if (run == 3456)  thetadeg = 25.7;
  else if (run == 3457)  thetadeg = 25.3;
  else if (run == 3458)  thetadeg = 24.9;
  else if (run == 3459)  thetadeg = 24.5;

  else if (run == 3460)  thetadeg = 24.1;

  else if (run == 3463)  thetadeg = 23.2;
  else if (run == 3464)  thetadeg = 22.8;
  else if (run == 3465)  thetadeg = 22.4;
  else if (run == 3466)  thetadeg = 22.0;
  else if (run == 3467)  thetadeg = 21.6;
  else if (run == 3468)  thetadeg = 21.2;
  else if (run == 3469)  thetadeg = 20.8;

  else if (run == 3470)  thetadeg = 20.4;
  else if (run == 3471)  thetadeg = 20.1;
  else if (run == 3472)  thetadeg = 19.7;
  else if (run == 3473)  thetadeg = 19.3;
  else if (run == 3474)  thetadeg = 18.9;
  else if (run == 3475)  thetadeg = 18.5;
  else if (run == 3476)  thetadeg = 18.2;
  else if (run == 3477)  thetadeg = 18.1;
  else if (run == 3478)  thetadeg = 17.7;


  else if (run == 3480)  thetadeg = 17.5;
  else if (run == 3481)  thetadeg = 17.1;
  else if (run == 3482)  thetadeg = 16.7;
  else if (run == 3483)  thetadeg = 16.3;
  else if (run == 3484)  thetadeg = 16.0;
  else if (run == 3485)  thetadeg = 15.6;
  else if (run == 3486)  thetadeg = 15.3;
  else if (run == 3487)  thetadeg = 15.0;
  else if (run == 3488)  thetadeg = 14.8;
  else if (run == 3489)  thetadeg = 14.8;
  
  else
  {
    *fLog << warn << "MPedestalWorkaround : no zenith angle for run number = "
          << run << endl;
    thetadeg = 90.0;
  }

  thetarad = thetadeg / kRad2Deg;
  fMcEvt->SetTelescopeTheta(thetarad);
  
  return kTRUE;
}

Int_t MPedestalWorkaround::PostProcess()
{
    return kTRUE;
}







