#ifndef MARS_MCalibrationChargePix
#define MARS_MCalibrationChargePix

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

class MBadPixelsPix;
class MCalibrationChargePix : public MParContainer
{
private:

  static const Float_t gkConversionHiLo;         // The default conversion factor HI gain - Lo Gain
  static const Float_t gkConversionHiLoErr;      // The error of the default conversion factor
  
  static const Float_t gkElectronicPedRms;       // The pure electronic component of the RMS
  static const Float_t gkElectronicPedRmsErr;    // The error of the pure electronic component of the RMS
  static const Float_t gkFFactor;                // The laboratory F-factor of the PMTs
  static const Float_t gkFFactorErr;             // The laboratory F-factor Error of the PMTs


  static const Float_t fgPheFFactorMethodLimit;  // The default minimum number of Photo-electrons for a pixel to be accepted.
  
  Float_t fElectronicPedRms;        // The pure electronic component of the RMS
  Float_t fElectronicPedRmsVar;     // The error of the pure electronic component of the RMS

  Float_t fPheFFactorMethodLimit;   // The minimum number of Photo-electrons for a pixel to be accepted.
  
  Int_t   fPixId;                   // the pixel Id

  UInt_t  fFlags;                   // Flag for the set bits
 
  Float_t fHiGainMeanCharge;        // The mean reduced charge after the fit
  Float_t fHiGainMeanChargeVar;     // The error of reduced mean charge after the fit
  Float_t fHiGainSigmaCharge;       // The sigma of the mean charge after the fit
  Float_t fHiGainSigmaChargeVar;    // The error of the sigma of the mean charge after the fit
  Float_t fHiGainChargeProb;        // The probability of the fit function 

  Float_t fLoGainMeanCharge;        // The mean reduced charge after the fit
  Float_t fLoGainMeanChargeVar;     // The error of reduced mean charge after the fit
  Float_t fLoGainSigmaCharge;       // The sigma of the mean charge after the fit
  Float_t fLoGainSigmaChargeVar;    // The error of the sigma of the mean charge after the fit
  Float_t fLoGainChargeProb;        // The probability of the fit function 

  Float_t fRSigmaCharge;            // The reduced squares of sigmas after the fit
  Float_t fRSigmaChargeVar;         // The reduced squares of sigmas after the fit  

  Float_t fPed;                     // The mean pedestal (from MPedestalPix) times number of FADC slices
  Float_t fPedVar;                  // The error of the pedestal 
  Float_t fPedRms;                  // The pedestal  RMS (from MPedestalPix) times sqrt of number of FADC slices

  Float_t fLoGainPedRms;            // The pedestal  RMS of the low gain
  Float_t fLoGainPedRmsVar;         // The pedestal  RMS Variance of the low gain

  Float_t fAbsTimeMean;             // The mean absolute arrival time
  Float_t fAbsTimeRms;              // The rms of the mean absolute arrival time
  
  Float_t fPheFFactorMethod;        // The number of Phe's calculated (F-factor method)
  Float_t fPheFFactorMethodVar;     // The error on the number of Phe's calculated (F-factor method)

  Float_t fMeanConversionFFactorMethod;     // The conversion factor to Phe's (F-factor method)
  Float_t fMeanConversionBlindPixelMethod;  // The conversion factor to Ph's (Blind Pixel method)
  Float_t fMeanConversionPINDiodeMethod;    // The conversion factor to Ph's (PIN Diode method)
  Float_t fMeanConversionCombinedMethod;    // The conversion factor to Ph's (all methods combined)

  Float_t fConversionFFactorMethodVar;      // The error of the conversion factor to Phe's (F-factor method)
  Float_t fConversionBlindPixelMethodVar;   // The error of the conversion factor to Ph's (Blind Pixel method)
  Float_t fConversionPINDiodeMethodVar;     // The error of the conversion factor to Ph's (PIN Diode method)
  Float_t fConversionCombinedMethodVar;     // The error of the conversion factor to Ph's (all methods combined)

  Float_t fSigmaConversionFFactorMethod;    // The sigma of conversion factor to Phe's (F-factor method)
  Float_t fSigmaConversionBlindPixelMethod; // The conversion factor to Ph's (Blind Pixel method)
  Float_t fSigmaConversionPINDiodeMethod;   // The conversion factor to Ph's (PIN Diode method)
  Float_t fSigmaConversionCombinedMethod;   // The conversion factor to Ph's (all methods combined)
 
  Float_t fTotalFFactorFFactorMethod;       // The total F-Factor to Ph's (F-factor method)
  Float_t fTotalFFactorBlindPixelMethod;    // The total F-Factor to Ph's (Blind Pixel method)
  Float_t fTotalFFactorPINDiodeMethod;      // The total F-Factor to Ph's (PIN Diode method)
  Float_t fTotalFFactorCombinedMethod;      // The total F-Factor to Ph's (all methods combined)
 
  Float_t fTotalFFactorFFactorMethodVar;    // The variance of the total F-Factor to Ph's (F-factor method)
  Float_t fTotalFFactorBlindPixelMethodVar; // The variance of the total F-Factor to Ph's (Blind Pixel method)
  Float_t fTotalFFactorPINDiodeMethodVar;   // The variance of the total F-Factor to Ph's (PIN Diode method)
  Float_t fTotalFFactorCombinedMethodVar;   // The variance of the total F-Factor to Ph's (all methods combined)
 
  Float_t fTotalFFactor;                    // The F-Factor of the total readout system (Sigma(out)/mean(out)*Mean(in)/sigma(in)
  Float_t fTotalFFactorVar;                 // The variance on the F-Factor of the total readout system
  
  Float_t fConversionHiLo;                  // The conversion factor between Hi Gain and Lo Gain  
  Float_t fConversionHiLoVar;               // The error of the conversion factor between Hi Gain and Lo Gain  

  Float_t fHiGainNumPickup;
  Float_t fLoGainNumPickup;

  enum  { kHiGainSaturation, kExcluded, 
          kBlindPixelMethodValid, kFFactorMethodValid, 
	  kPINDiodeMethodValid, kCombinedMethodValid };

public:

  MCalibrationChargePix(const char *name=NULL, const char *title=NULL);
  ~MCalibrationChargePix() {}
  
  void Clear(Option_t *o="");

  // Setter
  void SetPixId( const Int_t i )                                    { fPixId = i; }

  void SetPedestal(const Float_t ped, const Float_t pedrms, const Float_t pederr);

  void SetConversionHiLo(     const Float_t c = gkConversionHiLo)       { fConversionHiLo      = c;    }
  void SetConversionHiLoErr(  const Float_t e = gkConversionHiLoErr)    { fConversionHiLoVar   = e*e;    }

  void SetPheFFactorMethodLimit ( const Float_t f=fgPheFFactorMethodLimit  ) { fPheFFactorMethodLimit  = f;   }
  
  // Charges
  void SetHiGainMeanCharge      ( const Float_t f ) { fHiGainMeanCharge      = f; }
  void SetHiGainMeanChargeErr   ( const Float_t f ) { fHiGainMeanChargeVar   = f*f; }
  void SetHiGainChargeProb      ( const Float_t f ) { fHiGainChargeProb      = f; }
  void SetHiGainSigmaCharge     ( const Float_t f ) { fHiGainSigmaCharge     = f; }
  void SetHiGainSigmaChargeErr  ( const Float_t f ) { fHiGainSigmaChargeVar  = f*f; }

  void SetLoGainMeanCharge      ( const Float_t f ) { fLoGainMeanCharge      = f; }
  void SetLoGainMeanChargeErr   ( const Float_t f ) { fLoGainMeanChargeVar   = f*f; }
  void SetLoGainChargeProb      ( const Float_t f ) { fLoGainChargeProb      = f; }
  void SetLoGainSigmaCharge     ( const Float_t f ) { fLoGainSigmaCharge     = f; }
  void SetLoGainSigmaChargeErr  ( const Float_t f ) { fLoGainSigmaChargeVar  = f*f; }

  void SetMeanCharge            ( const Float_t f );                
  void SetMeanChargeErr         ( const Float_t f );             
  void SetSigmaCharge           ( const Float_t f );               
  void SetSigmaChargeErr        ( const Float_t f );            

  void SetHiGainNumPickup       ( const Float_t   f ) { fHiGainNumPickup       = f; }
  void SetLoGainNumPickup       ( const Float_t   f ) { fLoGainNumPickup       = f; }

  // Times
  void SetAbsTimeMean           ( const Float_t f ) { fAbsTimeMean           = f; }
  void SetAbsTimeRms            ( const Float_t f ) { fAbsTimeRms            = f; }

  // Conversion Factors
  void SetConversionFFactorMethod   ( Float_t c, Float_t err, Float_t sig );
  void SetConversionBlindPixelMethod( Float_t c, Float_t err, Float_t sig );
  void SetConversionPINDiodeMethod  ( Float_t c, Float_t err, Float_t sig );
  void SetConversionCombinedMethod  ( Float_t c, Float_t err, Float_t sig );

  void SetTotalFFactorFFactorMethod   ( const Float_t f)  { fTotalFFactorFFactorMethod = f; }
  void SetTotalFFactorBlindPixelMethod ( const Float_t f)  { fTotalFFactorBlindPixelMethod = f; }
  void SetTotalFFactorPINDiodeMethod   ( const Float_t f)  { fTotalFFactorPINDiodeMethod = f; }
  
  void SetTotalFFactorFFactorMethodErr   ( const Float_t f)  { fTotalFFactorFFactorMethodVar = f*f; }
  void SetTotalFFactorBlindPixelMethodErr ( const Float_t f)  { fTotalFFactorBlindPixelMethodVar = f*f; }
  void SetTotalFFactorPINDiodeMethodErr   ( const Float_t f)  { fTotalFFactorPINDiodeMethodVar = f*f; }
  
  // Bit Setters
  void SetHiGainSaturation    (  const Bool_t b = kTRUE );
  void SetExcluded            (  const Bool_t b = kTRUE );
  void SetBlindPixelMethodValid( const Bool_t b = kTRUE );
  void SetFFactorMethodValid  (  const Bool_t b = kTRUE );
  void SetPINDiodeMethodValid (  const Bool_t b = kTRUE );
  void SetCombinedMethodValid (  const Bool_t b = kTRUE );

  // Charges
  Float_t GetHiGainMeanCharge()          const { return fHiGainMeanCharge;      }
  Float_t GetHiGainMeanChargeErr()       const;
  Float_t GetHiGainChargeProb()          const { return fHiGainChargeProb;      }    
  Float_t GetHiGainSigmaCharge()         const { return fHiGainSigmaCharge;     }
  Float_t GetHiGainSigmaChargeErr()      const;

  Float_t GetLoGainMeanCharge()          const;
  Float_t GetLoGainMeanChargeErr()       const;
  Float_t GetLoGainChargeProb()          const { return fLoGainChargeProb;      }    
  Float_t GetLoGainSigmaCharge()         const;
  Float_t GetLoGainSigmaChargeErr()      const;

  Float_t GetMeanCharge()                const;
  Float_t GetMeanChargeErr()             const;
  Float_t GetChargeProb()                const;
  Float_t GetSigmaCharge()               const;
  Float_t GetSigmaChargeErr()            const;

  Float_t GetRSigmaCharge()              const;
  Float_t GetRSigmaChargeErr()           const;

  Float_t GetHiGainNumPickup()           const { return fHiGainNumPickup; }
  Float_t GetLoGainNumPickup()           const { return fLoGainNumPickup; }

  Float_t GetAbsTimeMean()               const { return fAbsTimeMean;     }
  Float_t GetAbsTimeRms()                const { return fAbsTimeRms;      }
  
  // Conversion Factors
  Float_t GetConversionHiLo()            const  { return fConversionHiLo;    }
  Float_t GetConversionHiLoErr()         const;

  Float_t GetMeanConversionBlindPixelMethod()  const { return fMeanConversionBlindPixelMethod  ; }
  Float_t GetConversionBlindPixelMethodErr()   const;
  Float_t GetSigmaConversionBlindPixelMethod() const { return fSigmaConversionBlindPixelMethod ; }

  Float_t GetMeanConversionFFactorMethod()     const { return fMeanConversionFFactorMethod;      }
  Float_t GetConversionFFactorMethodErr()      const;
  Float_t GetSigmaConversionFFactorMethod()    const { return fSigmaConversionFFactorMethod;     }

  Float_t GetMeanConversionPINDiodeMethod()    const { return fMeanConversionPINDiodeMethod ;    }
  Float_t GetConversionPINDiodeMethodErr()     const;
  Float_t GetSigmaConversionPINDiodeMethod()   const { return fSigmaConversionPINDiodeMethod ;   }

  Float_t GetMeanConversionCombinedMethod()    const { return fMeanConversionCombinedMethod ;    }
  Float_t GetConversionCombinedMethodErr()     const;
  Float_t GetSigmaConversionCombinedMethod()   const { return fSigmaConversionCombinedMethod ;   }

  Float_t GetPheFFactorMethod()                const { return fPheFFactorMethod;                 }    
  Float_t GetPheFFactorMethodErr()             const;

  Int_t   GetPixId()                           const { return fPixId;                            }

  Float_t GetPed()                             const { return fPed;                              }
  Float_t GetPedErr()                          const;
  Float_t GetPedRms()                          const;
  Float_t GetPedRmsErr()                       const;

  Float_t GetTotalFFactorFFactorMethod()       const { return fTotalFFactorFFactorMethod;        }
  Float_t GetTotalFFactorFFactorMethodErr()    const;
  
  Float_t GetTotalFFactorBlindPixelMethod()    const { return fTotalFFactorBlindPixelMethod;     }
  Float_t GetTotalFFactorBlindPixelMethodErr() const;
  
  Float_t GetTotalFFactorPINDiodeMethod()      const { return fTotalFFactorPINDiodeMethod;       }
  Float_t GetTotalFFactorPINDiodeMethodErr()   const;

  Float_t GetTotalFFactorCombinedMethod()      const { return fTotalFFactorCombinedMethod;       }
  Float_t GetTotalFFactorCombinedMethodErr()   const;
  
  Bool_t IsExcluded()                          const;
  Bool_t IsHiGainSaturation()                  const;
  Bool_t IsBlindPixelMethodValid()             const;
  Bool_t IsPINDiodeMethodValid()               const;
  Bool_t IsFFactorMethodValid()                const;
  Bool_t IsCombinedMethodValid()               const;

  // Miscellaneous
  void  CalcLoGainPedestal(const Float_t logainsamples);
  Bool_t CalcReducedSigma();
  Bool_t CalcFFactorMethod();

  ClassDef(MCalibrationChargePix, 1)	// Container for Calibration of one pixel
};

#endif

