/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                               
// MCalibrationChargeBlindPix    
//
// Storage container of the fit results of the Blind Pixel signal 
// (from MHCalibrationChargeBlindPix). 
//
// The Flux is calculated in photons per mm^2 in the camera plane. 
//
// Currently, the following numbers are implemented:
// - gkBlindPixelArea: 100 mm^2
// - Average QE of Blind Pixel: 
//    gkBlindPixelQEGreen: 0.154
//    gkBlindPixelQEBlue : 0.226
//    gkBlindPixelQEUV   : 0.247
//    gkBlindPixelQECT1  : 0.247
// - Average QE Error of Blind Pixel: 
//    gkBlindPixelQEGreenErr: 0.015;
//    gkBlindPixelQEBlueErr : 0.02;
//    gkBlindPixelQEUVErr   : 0.02;
//    gkBlindPixelQECT1Err  : 0.02;
// - Attenuation factor Blind Pixel:
//    gkBlindPixelAttGreen :  1.97;
//    gkBlindPixelAttBlue  :  1.96;
//    gkBlindPixelAttUV    :  1.95;
//    gkBlindPixelAttCT1   :  1.95;
//
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationChargeBlindPix.h"

#include <TH1.h>

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MCalibrationChargeBlindPix);

using namespace std;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelArea       = 100;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelAttGreen   = 1.97;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelAttBlue    = 1.96;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelAttUV      = 1.95;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelAttCT1     = 1.95;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQEGreen    = 0.154;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQEBlue     = 0.226;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQEUV       = 0.247;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQECT1      = 0.247;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQEGreenErr = 0.015;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQEBlueErr  = 0.02;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQEUVErr    = 0.02;
const Float_t MCalibrationChargeBlindPix::gkBlindPixelQECT1Err   = 0.02;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
// Calls:
// - Clear()
//
MCalibrationChargeBlindPix::MCalibrationChargeBlindPix(const char *name, const char *title)
{

  fName  = name  ? name  : "MCalibrationChargeBlindPix";
  fTitle = title ? title : "Container of the fit results of the blind pixel";

  Clear();
}


// ------------------------------------------------------------------------
//
// Sets:
// - all flags to kFALSE
// - all variables to -1.
//
// Calls: 
// - MCalibrationChargePix::Clear()
//
void MCalibrationChargeBlindPix::Clear(Option_t *o)
{

  fFluxInsidePlexiglass    = -1.;
  fFluxInsidePlexiglassVar = -1.;
  fLambda                  = -1.;
  fLambdaCheck             = -1.;
  fLambdaVar               = -1.;
  fMu0                     = -1.;
  fMu0Err                  = -1.;
  fMu1                     = -1.;
  fMu1Err                  = -1.;
  fSigma0                  = -1.;
  fSigma0Err               = -1.;
  fSigma1                  = -1.;
  fSigma1Err               = -1.;

  SetOscillating                   ( kFALSE );
  SetExcluded                      ( kFALSE );
  SetChargeFitValid                ( kFALSE );
  SetPedestalFitOK                 ( kFALSE );
  SetSinglePheFitOK                ( kFALSE );
  SetFluxInsidePlexiglassAvailable ( kFALSE );
  
  MCalibrationChargePix::Clear();
}

void  MCalibrationChargeBlindPix::SetFluxInsidePlexiglassAvailable( const Bool_t b)
{
    b ? SETBIT(fFlags,kFluxInsidePlexiglassAvailable) : CLRBIT(fFlags,kFluxInsidePlexiglassAvailable);
}

// --------------------------------------------------------------------------
//
// Set the Oscillating Bit from outside 
//
void  MCalibrationChargeBlindPix::SetOscillating( const Bool_t b)
{
    b ? SETBIT(fFlags,kOscillating) : CLRBIT(fFlags,kOscillating);
}

// --------------------------------------------------------------------------
//
// Set the ChargeFitValid Bit from outside 
//
void  MCalibrationChargeBlindPix::SetChargeFitValid( const Bool_t b)
{
    b ? SETBIT(fFlags,kChargeFitValid) : CLRBIT(fFlags,kChargeFitValid);
}

// --------------------------------------------------------------------------
//
// Set the PedestalFitValid Bit from outside 
//
void  MCalibrationChargeBlindPix::SetPedestalFitOK( const Bool_t b)
{
    b ? SETBIT(fFlags,kPedestalFitOK) : CLRBIT(fFlags,kPedestalFitOK);
}

// --------------------------------------------------------------------------
//
// Set the SinglePheFitValid Bit from outside 
//
void  MCalibrationChargeBlindPix::SetSinglePheFitOK( const Bool_t b)
{
    b ? SETBIT(fFlags,kSinglePheFitOK) : CLRBIT(fFlags,kSinglePheFitOK);
}

// --------------------------------------------------------------------------
//
// Return -1 if fFluxInsidePlexiglassVar is smaller than 0.
// Return square root of fFluxInsidePlexiglassVar
// 
Float_t MCalibrationChargeBlindPix::GetFluxInsidePlexiglassErr() const
{
  if (fFluxInsidePlexiglassVar < 0.)
    return -1.;
  
  return TMath::Sqrt(fFluxInsidePlexiglassVar);
}

// --------------------------------------------------------------------------
//
// Return -1 if fLambdaVar is smaller than 0.
// Return square root of fLambdaVar
// 
Float_t MCalibrationChargeBlindPix::GetLambdaErr() const
{
  if (fLambdaVar < 0.)
    return -1.;
  
  return TMath::Sqrt(fLambdaVar);
}

// --------------------------------------------------------------------------
//
// Test bit kChargeFitValid
//
Bool_t MCalibrationChargeBlindPix::IsChargeFitValid()  const 
{
    return TESTBIT(fFlags,kChargeFitValid);
}

// --------------------------------------------------------------------------
//
// Test bit kOscillating
//
Bool_t MCalibrationChargeBlindPix::IsOscillating()  const 
{
    return TESTBIT(fFlags,kOscillating);
}

// --------------------------------------------------------------------------
//
// Test bit kPedestalFitValid
//
Bool_t MCalibrationChargeBlindPix::IsPedestalFitOK()  const 
{
    return TESTBIT(fFlags,kPedestalFitOK);
}

// --------------------------------------------------------------------------
//
// Test bit kSinglePheFitValid
//
Bool_t MCalibrationChargeBlindPix::IsSinglePheFitOK()  const 
{
    return TESTBIT(fFlags,kSinglePheFitOK);
}

// --------------------------------------------------------------------------
//
// Test bit kFluxInsidePlexiglassAvailable
//
Bool_t  MCalibrationChargeBlindPix::IsFluxInsidePlexiglassAvailable()   const
{
  return TESTBIT(fFlags,kFluxInsidePlexiglassAvailable);
}


// --------------------------------------------------------------------------
//
// Return kFALSE if IsChargeFitValid() is kFALSE
//
// Calculate fFluxInsidePlexiglass with the formula:
// - fFluxInsidePlexiglass    = fLambda * gkBlindPixelArea / gkBlindPixelQE * 10**gkBlindPixelAtt
// - fFluxInsidePlexiglassVar = sqrt( fLambdaVar / ( fLambda * fLambda ) 
//                             + ( gkBlindPixelQEErr * gkBlindPixelQEErr / gkBlindPixelQE / gkBlindPixelQE )
//                               ) * fFluxInsidePlexiglass * * fFluxInsidePlexiglass
//
// If the fFluxInsidePlexiglass is smaller than 0., return kFALSE
// If the Variance is smaller than 0., return kFALSE
//
// SetFluxInsidePlexiglassAvailable() and return kTRUE
//
Bool_t MCalibrationChargeBlindPix::CalcFluxInsidePlexiglass()
{

  if (IsChargeFitValid())
    return kFALSE;
  

  //
  // Start calculation of number of photons 
  // The blind pixel has exactly 100 mm^2 area (with negligible error), 
  //
  fFluxInsidePlexiglass      = fLambda*gkBlindPixelArea;

  //
  // Start calculation of number of photons relative Variance 
  //
  const Float_t lambdaRelVar = fLambdaVar / ( fLambda * fLambda );
  fFluxInsidePlexiglassVar   = lambdaRelVar;

  switch (fColor)
    {
    case kGREEN:
      fFluxInsidePlexiglass    /=  gkBlindPixelQEGreen;   
      fFluxInsidePlexiglassVar +=  gkBlindPixelQEGreenErr * gkBlindPixelQEGreenErr
                               / ( gkBlindPixelQEGreen    * gkBlindPixelQEGreen   );   

      fFluxInsidePlexiglass  *= TMath::Power(10,gkBlindPixelAttGreen); // correct for absorption 
      // attenuation has negligible error
      break;
    case kBLUE:
      fFluxInsidePlexiglass    /=  gkBlindPixelQEBlue;   
      fFluxInsidePlexiglassVar +=  gkBlindPixelQEBlueErr * gkBlindPixelQEBlueErr
                               / ( gkBlindPixelQEBlue    * gkBlindPixelQEBlue    );   

      fFluxInsidePlexiglass *= TMath::Power(10,gkBlindPixelAttBlue); // correct for absorption 
      // attenuation has negligible error
      break;
    case kUV:
      fFluxInsidePlexiglass    /=  gkBlindPixelQEUV;   
      fFluxInsidePlexiglassVar +=  gkBlindPixelQEUVErr* gkBlindPixelQEUVErr
                               / ( gkBlindPixelQEUV   * gkBlindPixelQEUV    );   

      fFluxInsidePlexiglass *= TMath::Power(10,gkBlindPixelAttUV); // correct for absorption 
      // attenuation has negligible error
      break;
    case kCT1:
    default:
      fFluxInsidePlexiglass    /=  gkBlindPixelQECT1;   
      fFluxInsidePlexiglassVar +=  gkBlindPixelQECT1Err * gkBlindPixelQECT1Err
                               / ( gkBlindPixelQECT1    * gkBlindPixelQECT1    );   

      fFluxInsidePlexiglass *= TMath::Power(10,gkBlindPixelAttCT1); // correct for absorption 
      // attenuation has negligible error
      break;
    }

  *fLog << inf << endl;
  *fLog << inf << " Photon flux [ph/mm^2] inside Plexiglass: " 
        << fFluxInsidePlexiglass << endl;

  if (fFluxInsidePlexiglass < 0.)
      return kFALSE;

  if (fFluxInsidePlexiglassVar < 0.)
      return kFALSE;

  SetFluxInsidePlexiglassAvailable(kTRUE);  

  //
  // Finish calculation of errors -> convert from relative variance to absolute variance
  //
  fFluxInsidePlexiglassVar *= fFluxInsidePlexiglass * fFluxInsidePlexiglass;

  *fLog << inf << " Error on photon flux [ph/mm^2] inside Plexiglass: " 
        << GetFluxInsidePlexiglass() << endl;
  *fLog << inf << endl;

  return kTRUE;
}










