/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.

! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                               
// MCalibrationQECam                                               
//                                                               
// Storage container for the calibrated Quantum Efficiency of the whole camera.
//
// Individual pixels have to be cast when retrieved e.g.:
// MCalibrationQEPix &avpix = (MCalibrationQEPix&)(*fQECam)[i]
// 
// The following "calibration" constants can be retrieved from each pixel:
// - GetQE   ( const PulserColor_t color ): The mean quantum 
//   efficiency obtained with the calibration pulser color (e.g. kGREEN, kBLUE, kUV)
// - GetQEErr( const PulserColor_t color ): The uncertainty 
//   of the mean quantum efficiency.
//
// Averaged values over one whole area index (e.g. inner or outer pixels for 
// the MAGIC camera), can be retrieved via: 
// MCalibrationRelTimePix &avpix = (MCalibrationRelTimePix&)fRelCam->GetAverageArea(i)
//
// Averaged values over one whole camera sector can be retrieved via: 
// MCalibrationRelTimePix &avpix = (MCalibrationRelTimePix&)fRelCam->GetAverageSector(i)
//
// Note the averageing has been done at the end of the calculation and NOT on an 
// event-by-event basis (like in other classes deriving from MCalibrationCam). 
//
// See also: MHCalibrationChargePix, MHCalibrationChargeCam, MCalibrationChargeCalc
//                                                                         
// The calculated values (types of GetPixelContent) are:
// 
// 0: Mean Quantum Efficiency of the color: kCT1
// 1: Error of the Mean Quantum Efficiency of the color: kCT1
// 2: Mean Quantum Efficiency of the color: kGREEN
// 3: Error of the Mean Quantum Efficiency of the color: kGreen
// 4: Mean Quantum Efficiency of the color: kBLUE
// 5: Error of the Mean Quantum Efficiency of the color: kBlue
// 6: Mean Quantum Efficiency of the color: kUV
// 7: Error of the Mean Quantum Efficiency of the color: kUV
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationQECam.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MCalibrationQEPix.h"

ClassImp(MCalibrationQECam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Creates a TClonesArray of MCalibrationQEPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel. Later, a call to MCalibrationQECam::InitSize() 
// has to be performed (in MGeomApply). 
//
MCalibrationQECam::MCalibrationQECam(const char *name, const char *title)
{
    fName  = name  ? name  : "MCalibrationQECam";
    fTitle = title ? title : "Storage container for the calibrated Quantrum Efficiency of the camera";

    fPixels           = new TClonesArray("MCalibrationQEPix",1);
    fAverageAreas     = new TClonesArray("MCalibrationQEPix",1);
    fAverageSectors   = new TClonesArray("MCalibrationQEPix",1);

}


// --------------------------------------------------------------------------
//
// Print first the well fitted pixels 
// and then the ones which are not FitValid
//
void MCalibrationQECam::Print(Option_t *o) const
{

  *fLog << all << GetDescriptor() << ":" << endl;
  int id = 0;
  
  TIter Next(fPixels);
  MCalibrationQEPix *pix;
  while ((pix=(MCalibrationQEPix*)Next()))
    {
      
      if (!pix->IsExcluded() && pix->IsValid()) 
	{
	    *fLog << all << "Pix " << pix->GetPixId() 
		<< ":  QE: "                   << pix->GetQE(kCT1)        << " +- " << pix->GetQEErr(kCT1) 
		<< endl;
          id++;
	}
    }
  
  *fLog << all << id << " succesful pixels :-))" << endl;
  id = 0;
  
  *fLog << all << endl;
  *fLog << all << "Pixels with errors:" << endl;
  *fLog << all << endl;
  
  TIter Next2(fPixels);
    while ((pix=(MCalibrationQEPix*)Next2()))
      {
        
        if (!pix->IsExcluded() && !pix->IsValid())
          {
	    *fLog << all << "Pix " << pix->GetPixId() 
		<< ":  QE: "                   << pix->GetQE(kCT1)        << " +- " << pix->GetQEErr(kCT1) 
		<< endl;
            id++;
          }
      }
    *fLog << all << id << " pixels with errors :-((" << endl;

    
  *fLog << all << endl;
  *fLog << all << "Excluded pixels:" << endl;
  *fLog << all << endl;
  
  id = 0;

  TIter Next4(fPixels);
  while ((pix=(MCalibrationQEPix*)Next4()))
  {
      if (pix->IsExcluded())
      {
	  *fLog << all << pix->GetPixId() << endl;
	  id++;
      }
  }
  *fLog << all << id << " Excluded pixels " << endl;
}

// --------------------------------------------------------------------
//                                                                         
// The calculated values (types) are:
// 
// 0: Mean Quantum Efficiency of the color: kCT1
// 1: Error of the Mean Quantum Efficiency of the color: kCT1
// 2: Mean Quantum Efficiency of the color: kGREEN
// 3: Error of the Mean Quantum Efficiency of the color: kGreen
// 4: Mean Quantum Efficiency of the color: kBLUE
// 5: Error of the Mean Quantum Efficiency of the color: kBlue
// 6: Mean Quantum Efficiency of the color: kUV
// 7: Error of the Mean Quantum Efficiency of the color: kUV
//
Bool_t MCalibrationQECam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (idx > GetSize())
    return kFALSE;

  MCalibrationQEPix &pix = (MCalibrationQEPix&)(*this)[idx];

  if (pix.IsExcluded())
    return kFALSE;

  switch (type)
    {
    case 0:
      val = pix.GetQE(kCT1);
      break;
    case 1:
      val = pix.GetQEErr(kCT1);
      break;
    case 2:
      val = pix.GetQE(kGREEN);
      break;
    case 3:
      val = pix.GetQEErr(kGREEN);
      break;
    case 4:
      val = pix.GetQE(kBLUE);
      break;
    case 5:
      val = pix.GetQEErr(kBLUE);
      break;
    case 6:
      val = pix.GetQE(kUV);
      break;
    case 7:
      val = pix.GetQEErr(kUV);
      break;
    default:
      return kFALSE;
    }
  return val!=-1.;
}

// --------------------------------------------------------------------------
//
// Not yet implemented
//
void MCalibrationQECam::DrawPixelContent(Int_t idx) const
{
  return;
}









