#ifndef MARS_MGeomCam
#define MARS_MGeomCam

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif
#ifndef ROOT_TObjArray
#include <TObjArray.h>
#endif
#ifndef ROOT_TArrayF
#include <TArrayF.h>
#endif

class TVector2;
class MGeomPix;

class MGeomCam : public MParContainer
{
private:
    UInt_t    fNumPixels;    // Number of pixels in this camera

    Float_t   fCamDist;      // [m] Average distance of the camera from the mirror
    Float_t   fConvMm2Deg;   // conversion factor to convert mm in the camera plain into degrees

    TObjArray fPixels;       // Array of singel pixels storing the geometry

    TArrayF   fMaxRadius;    // maximum radius of the part of the camera with the same pixel size (eg. for GUI layout)
    TArrayF   fMinRadius;    // minimum radius of the part of the camera with the same pixel size (eg. for GUI layout)    
    TArrayF   fPixRatio;     // Array storing the ratio between size of pixel 0 and pixel idx (for speed reasons)
    TArrayF   fPixRatioSqrt; // Array storing the square root ratio between size of pixel 0 and pixel idx (for speed reasons)

    Int_t     fNumSectors;   // Number of sectors
    Int_t     fNumAreas;     // Number of different pixel sizes

protected:

    void CalcMaxRadius();
    void CalcNumSectors();
    void CalcNumAreas();
    void InitOuterRing();
    void InitGeometry()
    {
        CalcNumSectors();
        CalcNumAreas();
        CalcMaxRadius();
        CalcPixRatio();
        InitOuterRing();
    }


public:
    MGeomCam();
    MGeomCam(UInt_t npix, Float_t dist, const char *name=NULL, const char *title=NULL);

    virtual TObject *Clone(const char *newname=NULL) const;

    void CalcPixRatio(); 
    // FIXME, workaround: this function is made public just to allow
    // the use of some camera files from the 0.7 beta version in which the 
    // array containing pixel ratios is not initialized.

    Float_t GetCameraDist() const { return fCamDist; }
    Float_t GetConvMm2Deg() const { return fConvMm2Deg; }

    UInt_t  GetNumPixels()  const { return fNumPixels; }

    Float_t GetMaxRadius(const Int_t i=-1) const;
    Float_t GetMinRadius(const Int_t i=-1) const;
    
    UInt_t  GetNumSectors()                const  { return fNumSectors; }
    UInt_t  GetNumAreas()                  const  { return fNumAreas; }
    Float_t GetPixRatio(UInt_t i)          const;
    Float_t GetPixRatioSqrt(UInt_t i)      const;

    MGeomPix &operator[](Int_t i);
    MGeomPix &operator[](Int_t i) const;

    Int_t GetPixelIdx(const TVector2 &v) const;
    Int_t GetPixelIdxDeg(const TVector2 &v) const;
    Int_t GetPixelIdxXY(Float_t x, Float_t y) const;
    Int_t GetPixelIdxXYdeg(Float_t x, Float_t y) const
    {
        return GetPixelIdxXY(x/fConvMm2Deg, y/fConvMm2Deg);
    }

    virtual void Print(Option_t *opt=NULL)   const;

    ClassDef(MGeomCam, 3)  // Geometry base class for the camera
};

#endif
